#!/usr/bin/env python3
"""
Configuration utilities for Abflug clients.

Provides functions to load API keys and server URLs from butter.json or abflug.json if available.
"""

import json
import os
from typing import Optional

# Default server URL if not found in config files
DEFAULT_SERVER_URL = "https://abflug.cloud"


def _load_config_file(filename: str) -> Optional[dict]:
    """
    Load a JSON config file from the same directory as this script.
    
    Args:
        filename: Name of the config file (e.g., "butter.json", "abflug.json")
    
    Returns:
        Dictionary with config data if found, None otherwise
    """
    base_dir = os.path.dirname(os.path.abspath(__file__))
    config_path = os.path.join(base_dir, filename)
    
    if not os.path.exists(config_path):
        return None
    
    try:
        with open(config_path, "r", encoding="utf-8") as f:
            return json.load(f)
    except Exception:
        # Silently fail if we can't read the file
        return None


def get_butter_api_key() -> Optional[str]:
    """
    Get API key from butter.json or abflug.json if it exists.
    
    Returns:
        API key string if found, None otherwise
    """
    # Try butter.json first, then abflug.json
    for config_file in ["butter.json", "abflug.json"]:
        config = _load_config_file(config_file)
        if config:
            api_key = config.get("api_key", "").strip()
            if api_key:
                return api_key
    
    return None

def get_debug_api_requests() -> bool:
    """
    Get debug API requests flag from butter.json or abflug.json if it exists.
    
    Returns:
        Debug API requests flag if found, False otherwise
    """
    for config_file in ["butter.json", "abflug.json"]:
        config = _load_config_file(config_file)
        if config:
            debug_api_requests = config.get("debug_api_requests", "False")
            if debug_api_requests.lower() == "true":
                return True
            elif debug_api_requests.lower() == "false":
                return False
    return False  # Default to False if not found
    

def get_server_url() -> str:
    """
    Get server URL from butter.json or abflug.json if it exists.
    
    Returns:
        Server URL string from config file, or DEFAULT_SERVER_URL if not found
    """
    # Try butter.json first, then abflug.json
    for config_file in ["butter.json", "abflug.json"]:
        config = _load_config_file(config_file)
        if config:
            server_url = config.get("server", "").strip()
            if server_url:
                return server_url
    
    return DEFAULT_SERVER_URL

