#!/usr/bin/env python3
"""
Butter-o-meter™ – simple landing rater built on the Abflug client platform.

Flow:
- Autodetect simulator FSWidget endpoint on the local network
- Ask user for API key (stored in butter.json) and landing airport ICAO
- Query Abflug API for airport altitude
- Read simulator data via FSWidget at 10 Hz using SimulatorDataSource
- Detect descent and landing phases
- After landing, compute a 0–100 butter score from vertical/ground-speed changes
- Show a meme-friendly summary and optionally post to Discord
"""

import argparse
import json
import os
import socket
import sys
import threading
import time
from dataclasses import dataclass
from datetime import datetime, timezone
from typing import Any, Dict, List, Optional, Tuple

import requests

from simulator_data_source import (
    SimulatorDataSource,
    DataSourceType,
    GPSData,
    FSWIDGET_PORT,
    FSWIDGET_REQUEST,
)

from discord_hooks_module import get_color_code


APP_NAME = "Butter-o-meter"
APP_VERSION = "1.0.0"
DEFAULT_SERVER_URL = "https://abflug.cloud"
CONFIG_FILENAME = "butter.json"


@dataclass
class ButterSample:
    """Single time-stamped sample used for landing analysis."""

    timestamp: float
    altitude_ft: float
    vertical_speed_fpm: float
    ground_speed_kts: float


@dataclass
class ButterResult:
    """Final landing analysis result."""

    score: int
    smoothness: int
    vertical_speed_fpm: float
    ground_speed_kts: float
    category: str
    emoji_line: str
    message_line: str


def butter_config_path() -> str:
    """Return the absolute path to butter.json next to this script."""
    base_dir = os.path.dirname(os.path.abspath(__file__))
    return os.path.join(base_dir, CONFIG_FILENAME)


def load_config() -> Dict[str, Any]:
    path = butter_config_path()
    if not os.path.exists(path):
        return {}
    try:
        with open(path, "r", encoding="utf-8") as f:
            return json.load(f)
    except Exception:
        # Corrupt config – start fresh but keep file around
        return {}


def save_config(cfg: Dict[str, Any]) -> None:
    path = butter_config_path()
    try:
        with open(path, "w", encoding="utf-8") as f:
            json.dump(cfg, f, indent=2)
    except Exception as e:
        print(f"Warning: could not save {CONFIG_FILENAME}: {e}")


def ensure_api_key_in_config(config: Dict[str, Any]) -> str:
    api_key = config.get("api_key", "").strip()
    if api_key:
        return api_key

    print("\nNo API key found in butter.json.")
    print("To get an API key, sign in at:")
    print("  https://abflug.cloud/login.html")
    print("Then paste your API key here.\n")
    api_key = input("Enter your Abflug API key: ").strip()
    if not api_key:
        print("No API key entered. Exiting.")
        sys.exit(1)

    config["api_key"] = api_key
    if "server" not in config:
        config["server"] = DEFAULT_SERVER_URL
    save_config(config)
    return api_key


def ensure_server_in_config(config: Dict[str, Any]) -> str:
    server = config.get("server", "").strip()
    if not server:
        server = DEFAULT_SERVER_URL
        config["server"] = server
        save_config(config)
    return server


def get_local_ip() -> str:
    """
    Get the primary local IP address used for outbound traffic.
    This avoids relying on hostname resolution which may return 127.0.0.1.
    """
    try:
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        # Connect to a public IP (no traffic actually sent) just to get routing info
        s.connect(("8.8.8.8", 80))
        ip = s.getsockname()[0]
        s.close()
        return ip
    except Exception:
        return "127.0.0.1"


def discover_fswidget_ip(timeout_per_host: float = 0.2) -> Optional[str]:
    """
    Autodetect FSWidget TCP endpoint on the local /24 network.

    - Detect local IPv4 address
    - Scan all addresses x.x.x.1–254 on FSWIDGET_PORT
    - Send FSWidget handshake and look for a valid FSWidget response
    """
    local_ip = get_local_ip()
    if local_ip.startswith("127."):
        print("Could not determine a non-loopback IP address; defaulting to localhost for FSWidget.")
        return "localhost"

    parts = local_ip.split(".")
    if len(parts) != 4:
        print(f"Unexpected local IP format '{local_ip}', using localhost for FSWidget.")
        return "localhost"

    base_prefix = ".".join(parts[:3])
    print(f"\nAutodetecting simulator on {base_prefix}.0/24 using FSWidget (port {FSWIDGET_PORT})...")

    def try_host(host: str) -> bool:
        try:
            sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            sock.settimeout(timeout_per_host)
            sock.connect((host, FSWIDGET_PORT))
            # Send FSWidget handshake
            sock.sendall(FSWIDGET_REQUEST.encode("utf-8"))
            data = sock.recv(128)
            sock.close()
            msg = data.decode("utf-8", errors="ignore").strip()
            # FSWidget messages are pipe-separated with at least 8 fields
            if msg.count("|") >= 7:
                print(f"Found FSWidget server at {host}")
                return True
        except Exception:
            return False
        return False

    # Try the local host first – often the sim runs on the same machine
    if try_host(local_ip):
        return local_ip

    # Fallback: scan the /24
    for i in range(1, 255):
        candidate = f"{base_prefix}.{i}"
        if candidate == local_ip:
            continue
        if try_host(candidate):
            return candidate

    print("No FSWidget server found on the local /24 network.")
    return None


def get_airport_info(server_url: str, api_key: str, icao_code: str) -> Optional[Dict[str, Any]]:
    """
    Get airport information from Abflug server API (v2).
    """
    session = requests.Session()
    session.headers.update(
        {
            "X-API-Key": api_key,
            "Content-Type": "application/json",
        }
    )
    try:
        resp = session.get(
            f"{server_url.rstrip('/')}/api/airport",
            params={"icao_code": icao_code},
            timeout=5.0,
        )
        resp.raise_for_status()
        return resp.json()
    except requests.exceptions.RequestException as e:
        if hasattr(e, "response") and e.response is not None:
            if e.response.status_code == 404:
                print(f"Airport {icao_code} not found in server database.")
            else:
                print(
                    f"Error fetching airport info for {icao_code}: "
                    f"{e.response.status_code} - {e.response.text[:120]}"
                )
        else:
            print(f"Error fetching airport info for {icao_code}: {e}")
        return None


class LandingDetector:
    """
    Simplified state machine to detect the landing phase based on
    vertical speed and altitude relative to the destination airport.
    """

    def __init__(self, airport_altitude_ft: float):
        self.airport_altitude_ft = airport_altitude_ft
        self.state = "cruise"  # cruise -> descending -> landing -> landed
        self.last_sample_time: float = 0.0
        self.samples: List[ButterSample] = []
        self.touchdown_time: Optional[float] = None

    def add_sample(self, gps: GPSData, timestamp: Optional[float] = None) -> None:
        ts = timestamp if timestamp is not None else time.time()
        sample = ButterSample(
            timestamp=ts,
            altitude_ft=gps.altitude,
            vertical_speed_fpm=gps.vertical_speed,
            ground_speed_kts=gps.ground_speed,
        )
        print(f"{sample.timestamp}, {sample.altitude_ft}, {sample.vertical_speed_fpm}, {sample.ground_speed_kts}")
        self.samples.append(sample)
        # Keep last ~5 minutes just in case; we only really need 30+ seconds
        cutoff = ts - 300.0
        self.samples = [s for s in self.samples if s.timestamp >= cutoff]
        self.last_sample_time = ts
        self._update_state(sample)

    def _update_state(self, s: ButterSample) -> None:
        vs = s.vertical_speed_fpm
        alt = s.altitude_ft

        # thresholds
        cruise_vs_threshold = 100.0  # fpm
        descending_vs_threshold = -200.0  # fpm
        near_airport_margin_ft = 300.0

        if self.state in ("cruise", "unknown"):
            if vs < descending_vs_threshold:
                self.state = "descending"
        elif self.state == "descending":
            if alt <= self.airport_altitude_ft + near_airport_margin_ft and vs < 0:
                self.state = "landing"
        elif self.state == "landing":
            # Touchdown: vertical speed close to zero and altitude close to airport
            if (
                abs(vs) <= cruise_vs_threshold
                and abs(alt - self.airport_altitude_ft) <= near_airport_margin_ft
            ):
                self.state = "landed"
                #print(f"{s.timestamp} - touchdown detected")
                self.touchdown_time = s.timestamp

    def has_landed(self) -> bool:
        return self.state == "landed" and self.touchdown_time is not None

    def get_analysis_window(self) -> Tuple[List[ButterSample], Optional[float]]:
        """
        Return samples in the analysis window:
        - 10 seconds before touchdown up to when vertical speed is flat for 20 consecutive samples
        (captures VS oscillations during landing without including rollout phase)
        """
        if not self.has_landed():
            return [], None
        td = self.touchdown_time  # type: ignore[assignment]
        start = td - 10.0
        
        # Find the point after touchdown where vertical speed is flat for 20 consecutive samples
        # "Flat" means vertical speed is within ±50 fpm of zero
        post_td_samples = sorted([s for s in self.samples if s.timestamp >= td], key=lambda s: s.timestamp)
        end_time = td + 15.0  # Default: 15 seconds after touchdown (safety timeout)
        
        flat_vs_threshold = 1.0  # fpm - consider VS "flat" if within ±1 fpm
        consecutive_flat_count = 0
        required_flat_samples = 20
        
        for sample in post_td_samples:
            if abs(sample.vertical_speed_fpm) <= flat_vs_threshold:
                consecutive_flat_count += 1
                if consecutive_flat_count >= required_flat_samples:
                    # End window at the 20th consecutive flat sample (captures all oscillations before this)
                    end_time = sample.timestamp
                    print(f"Vertical speed flat for {required_flat_samples} consecutive samples at {end_time:.2f} (t+{end_time - td:.2f}s)")
                    break
            else:
                consecutive_flat_count = 0  # Reset counter if VS is not flat
        
        window = [s for s in self.samples if start <= s.timestamp <= end_time]
        return window, td


def compute_butter_score(window: List[ButterSample], touchdown_time: float) -> ButterResult:
    """
    Compute butter score from samples in the analysis window.

    We approximate "jerk" as the discrete change of vertical/ground speed per second
    and combine their peaks into a 0–100 score (higher is smoother).
    """
    if len(window) < 2:
        # Not enough data – fallback with neutral values
        return ButterResult(
            score=50,
            smoothness=50,
            vertical_speed_fpm=0.0,
            ground_speed_kts=0.0,
            category="INSUFFICIENT DATA",
            emoji_line="🤔",
            message_line="Not enough data to rate this landing",
        )

    # Sort by time to be safe
    window = sorted(window, key=lambda s: s.timestamp)

    max_dv_dt = 0.0  # fpm/s
    max_dg_dt = 0.0  # kts/s
    vs_at_touchdown = 0.0
    gs_at_touchdown = 0.0
    closest_td_sample: Optional[ButterSample] = None

    for i in range(1, len(window)):
        prev = window[i - 1]
        cur = window[i]
        dt = max(cur.timestamp - prev.timestamp, 1e-3)
        dv = cur.vertical_speed_fpm - prev.vertical_speed_fpm
        dg = cur.ground_speed_kts - prev.ground_speed_kts
        max_dv_dt = max(max_dv_dt, abs(dv) / dt)
        max_dg_dt = max(max_dg_dt, abs(dg) / dt)

        if closest_td_sample is None or abs(cur.timestamp - touchdown_time) < abs(
            closest_td_sample.timestamp - touchdown_time
        ):
            closest_td_sample = cur

    if closest_td_sample:
        vs_at_touchdown = closest_td_sample.vertical_speed_fpm
        gs_at_touchdown = closest_td_sample.ground_speed_kts

    # Normalize roughness: pick "very rough" reference values
    # These are heuristic and tuned for meme-worthiness, not physics accuracy.
    dv_ref = 800.0  # fpm/s – huge sudden change
    dg_ref = 30.0  # kts/s – big ground speed change
    dv_norm = min(max_dv_dt / dv_ref, 1.5)
    dg_norm = min(max_dg_dt / dg_ref, 1.5)

    # Combine with weights (vertical dominance)
    roughness = 0.7 * dv_norm + 0.3 * dg_norm
    roughness_clamped = max(0.0, min(roughness, 1.5))

    # Base score from roughness (0–100, where 0 is hull loss and 100 is silk smooth)
    base_score = int(round(100 * (1.0 - (roughness_clamped / 1.5))))

    # Small bonus/penalty based on actual vertical speed at touchdown:
    # -150 fpm is ideal, -300 fpm is okay, below -600 is bad.
    vs_penalty = 0
    if vs_at_touchdown < -600:
        vs_penalty = -25
    elif vs_at_touchdown < -400:
        vs_penalty = -15
    elif vs_at_touchdown < -250:
        vs_penalty = -5
    elif vs_at_touchdown > -120:
        # too flat can feel floaty; tiny penalty
        vs_penalty = -3

    final_score = max(0, min(100, base_score + vs_penalty))
    smoothness = final_score

    # Category + emojis
    if final_score >= 90:
        category = "SILK SMOOTH TOUCHDOWN"
        emoji_line = "🧈🧈🧈"
        message_line = "BUTTER LANDING!"
    elif final_score >= 75:
        category = "NICE AND SOFT"
        emoji_line = "🧈"
        message_line = "Butter-ish landing"
    elif final_score >= 60:
        category = "MEH LANDING"
        emoji_line = "😐"
        message_line = "Meh, it'll pass"
    elif final_score >= 45:
        category = "SPICY ARRIVAL"
        emoji_line = "😬😬😬"
        message_line = "That was… firm"
    elif final_score >= 30:
        category = "CALL THE AMBULANCE"
        emoji_line = "🚑"
        message_line = "Cabin crew, prepare for meme review"
    else:
        category = "HULL LOSS"
        emoji_line = "💥"
        message_line = "HULL LOSS – send the clip"

    return ButterResult(
        score=final_score,
        smoothness=smoothness,
        vertical_speed_fpm=vs_at_touchdown,
        ground_speed_kts=gs_at_touchdown,
        category=category,
        emoji_line=emoji_line,
        message_line=message_line,
    )


def append_landing_to_config(
    config: Dict[str, Any],
    airport_icao: str,
    result: ButterResult,
) -> None:
    """Append landing result to butter.json under 'landings'."""
    landings = config.get("landings")
    if not isinstance(landings, list):
        landings = []
    landings.append(
        {
            "timestamp_utc": datetime.now(timezone.utc).strftime("%Y-%m-%dT%H:%M:%SZ"),
            "airport": airport_icao.upper(),
            "score": result.score,
            "smoothness": result.smoothness,
            "vertical_speed_fpm": result.vertical_speed_fpm,
            "ground_speed_kts": result.ground_speed_kts,
            "category": result.category,
        }
    )
    config["landings"] = landings
    save_config(config)


def build_discord_payload(
    airport_icao: str,
    result: ButterResult,
) -> Dict[str, Any]:
    title = f"Butter-o-meter – {airport_icao.upper()} landing"
    description = (
        f"{result.emoji_line} {result.message_line}\n"
        f"**Butter score**: {result.score}/100\n"
        f"**Smoothness**: {result.smoothness}/100\n"
        f"**Vertical speed**: {result.vertical_speed_fpm:.0f} fpm\n"
        f"**Ground speed**: {result.ground_speed_kts:.0f} kts\n"
        f"**Verdict**: {result.category}\n"
        f"Powered by Abflug"
    )

    return {
        "content": "",
        "embeds": [
            {
                "title": title,
                "description": description,
                "color": get_color_code("Aqua"),
            }
        ],
    }


def maybe_send_discord(
    config: Dict[str, Any],
    airport_icao: str,
    result: ButterResult,
) -> None:
    webhook_url = config.get("discord_webhook", "").strip()
    if not webhook_url:
        # Ask user if they want to provide one just for this run
        choice = input("\nShare to Discord? (y/N): ").strip().lower()
        if choice != "y":
            return
        webhook_url = input("Paste Discord webhook URL (starts with https://discord.com/api/webhooks/): ").strip()
        if not webhook_url:
            print("No webhook entered. Skipping Discord.")
            return
        # Optionally persist
        save_choice = input("Save this webhook in butter.json for next time? (y/N): ").strip().lower()
        if save_choice == "y":
            config["discord_webhook"] = webhook_url
            save_config(config)
    else:
        choice = input("\nShare latest landing to Discord using saved webhook? (y/N): ").strip().lower()
        if choice != "y":
            return

    if not webhook_url.startswith("https://discord.com/api/webhooks/"):
        print("Invalid Discord webhook URL. Must start with https://discord.com/api/webhooks/")
        return

    payload = build_discord_payload(airport_icao, result)
    try:
        resp = requests.post(webhook_url, json=payload, timeout=5.0)
        if resp.status_code not in (200, 204):
            print(f"Error sending Discord webhook: {resp.status_code} - {resp.text[:200]}")
        else:
            print("Discord webhook sent successfully.")
    except Exception as e:
        print(f"Error sending Discord webhook: {e}")


def draw_console_ui(result: ButterResult, airport_icao: str) -> None:
    """Render a simple text UI with the butter-o-meter look."""
    title = "BUTTER-O-METER™"
    powered = "Powered by Abflug"
    smooth_line = f"Smoothness: {result.smoothness}/100"
    vs_line = f"Vertical Speed: {result.vertical_speed_fpm:.0f} fpm"
    gs_line = f"Ground Speed: {result.ground_speed_kts:.0f} kts"
    cat_line = result.category
    airport_line = f"Airport: {airport_icao.upper()}"

    lines = [
        title,
        "",
        result.emoji_line,
        "",
        result.message_line,
        "",
        smooth_line,
        vs_line,
        gs_line,
        "",
        cat_line,
        airport_line,
        "",
        "[Share to Discord]",  # purely decorative; actual send is confirmed separately
        "",
        powered,
    ]

    width = max(len(l) for l in lines) + 4
    top = "╔" + "═" * (width - 2) + "╗"
    sep = "╠" + "═" * (width - 2) + "╣"
    bottom = "╚" + "═" * (width - 2) + "╝"

    def pad(line: str) -> str:
        return "║ " + line.ljust(width - 4) + " ║"

    print("\n" + top)
    print(pad(title))
    print(sep)
    for line in lines[2:]:
        print(pad(line))
    print(bottom)


def run_butter_meter() -> int:
    print(f"{APP_NAME} v{APP_VERSION}")

    # 1. Load config and ensure API key & server
    config = load_config()
    api_key = ensure_api_key_in_config(config)
    server_url = ensure_server_in_config(config)

    # 2. Ask for arrival airport
    airport_icao = input("\nEnter landing airport ICAO (e.g. LOWG): ").strip().upper()
    if not airport_icao:
        print("No airport entered. Exiting.")
        return 1

    # 3. Fetch airport altitude
    airport_info = get_airport_info(server_url, api_key, airport_icao)
    if airport_info:
        airport_altitude_ft = float(airport_info.get("elevation_ft", 0.0) or 0.0)
        print(f"Airport {airport_icao} elevation: {airport_altitude_ft:.0f} ft (from Abflug API)")
    else:
        airport_altitude_ft = 0.0
        print(f"Warning: using default airport elevation {airport_altitude_ft:.0f} ft")

    # 4. Autodetect FSWidget simulator endpoint
    fs_ip = discover_fswidget_ip()
    if not fs_ip:
        print("Could not find FSWidget simulator endpoint. Make sure your sim is running with FSWidget enabled.")
        return 1

    print(f"Using FSWidget IP: {fs_ip}")

    # 5. Start simulator data source (FSWidget) at 10 Hz
    data_source = SimulatorDataSource(
        source_type=DataSourceType.FSWIDGET,
        fswidget_ip=fs_ip,
        fswidget_port=FSWIDGET_PORT,
    )

    detector = LandingDetector(airport_altitude_ft=airport_altitude_ft)
    stop_event = threading.Event()

    def on_gps_update(gps: GPSData) -> None:
        detector.add_sample(gps)

    data_source.on_gps_update = on_gps_update

    print("\nStarting Butter-o-meter. Reading data from simulator...")
    data_source.start()

    try:
        touchdown_detected = False
        flat_vs_threshold = 50.0  # fpm - consider VS "flat" if within ±50 fpm
        required_flat_samples = 20
        consecutive_flat_count = 0
        
        while not stop_event.is_set():
            time.sleep(0.5)
            if detector.has_landed() and not touchdown_detected:
                print("\nTouchdown detected – continuing to collect data until VS is flat for 20 samples...")
                touchdown_detected = True
                # Continue collecting for up to 15 seconds after touchdown
                # to capture VS oscillations without including rollout
                touchdown_time = detector.touchdown_time
                timeout = time.time() + 15.0  # Max 15 seconds after touchdown
            
            # If touchdown detected, check if we should stop (VS flat for 20 samples or timeout)
            if touchdown_detected:
                # Check the most recent samples in the detector for flat VS
                if detector.samples:
                    # Look at the last 20 samples (or fewer if we don't have 20 yet)
                    recent_samples = detector.samples[-required_flat_samples:] if len(detector.samples) >= required_flat_samples else detector.samples
                    # Only check samples after touchdown
                    post_td_samples = [s for s in recent_samples if s.timestamp >= detector.touchdown_time]
                    
                    if len(post_td_samples) >= required_flat_samples:
                        # Check if the last 20 post-touchdown samples are all flat
                        all_flat = all(abs(s.vertical_speed_fpm) <= flat_vs_threshold for s in post_td_samples[-required_flat_samples:])
                        if all_flat:
                            print(f"Vertical speed flat for {required_flat_samples} consecutive samples – stopping data collection.")
                            break
                
                if time.time() >= timeout:
                    print("Timeout reached (15s after touchdown) – stopping data collection.")
                    break
    except KeyboardInterrupt:
        print("\nInterrupted by user.")
        stop_event.set()

    data_source.stop()

    if not detector.has_landed():
        print("No landing detected before stop. Exiting.")
        return 1

    window, td = detector.get_analysis_window()
    if not window or td is None:
        print("Insufficient data in landing window. Exiting.")
        return 1

    result = compute_butter_score(window, td)

    # 6. Print fancy console UI
    draw_console_ui(result, airport_icao)

    # 7. Append to butter.json
    append_landing_to_config(config, airport_icao, result)

    # 8. Optional Discord share
    maybe_send_discord(config, airport_icao, result)

    return 0


def main() -> int:
    parser = argparse.ArgumentParser(
        description="Butter-o-meter – rate your landings with Abflug data",
    )
    # Reserved for future CLI flags; currently interactive only
    _ = parser.parse_args()
    return run_butter_meter()


if __name__ == "__main__":
    sys.exit(main())


