#!/usr/bin/env python3
"""
Butter-o-meter™ GUI – simple landing rater built on the Abflug client platform.

Tkinter GUI version with the same functionality as butter_meter.py
"""

import json
import os
import socket
import sys
import threading
import time
import tkinter as tk
from tkinter import ttk, messagebox, simpledialog
from dataclasses import dataclass
from datetime import datetime, timezone
from typing import Any, Dict, List, Optional, Tuple

import requests

from simulator_data_source import (
    SimulatorDataSource,
    DataSourceType,
    GPSData,
    FSWIDGET_PORT,
    FSWIDGET_REQUEST,
)

from discord_hooks_module import get_color_code


APP_NAME = "Butter-o-meter"
APP_VERSION = "1.0.0"
DEFAULT_SERVER_URL = "https://abflug.cloud"
CONFIG_FILENAME = "butter.json"


@dataclass
class ButterSample:
    """Single time-stamped sample used for landing analysis."""

    timestamp: float
    altitude_ft: float
    vertical_speed_fpm: float
    ground_speed_kts: float


@dataclass
class ButterResult:
    """Final landing analysis result."""

    score: int
    smoothness: int
    vertical_speed_fpm: float
    ground_speed_kts: float
    category: str
    emoji_line: str
    message_line: str


def butter_config_path() -> str:
    """Return the absolute path to butter.json next to this script."""
    base_dir = os.path.dirname(os.path.abspath(__file__))
    return os.path.join(base_dir, CONFIG_FILENAME)


def load_config() -> Dict[str, Any]:
    path = butter_config_path()
    if not os.path.exists(path):
        return {}
    try:
        with open(path, "r", encoding="utf-8") as f:
            return json.load(f)
    except Exception:
        return {}


def save_config(cfg: Dict[str, Any]) -> None:
    path = butter_config_path()
    try:
        with open(path, "w", encoding="utf-8") as f:
            json.dump(cfg, f, indent=2)
    except Exception as e:
        pass  # Silently fail in GUI mode


def get_local_ip() -> str:
    """Get the primary local IP address used for outbound traffic."""
    try:
        s = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        s.connect(("8.8.8.8", 80))
        ip = s.getsockname()[0]
        s.close()
        return ip
    except Exception:
        return "127.0.0.1"


def discover_fswidget_ip(timeout_per_host: float = 0.2) -> Optional[str]:
    """Autodetect FSWidget TCP endpoint on the local /24 network."""
    local_ip = get_local_ip()
    if local_ip.startswith("127."):
        return "localhost"

    parts = local_ip.split(".")
    if len(parts) != 4:
        return "localhost"

    base_prefix = ".".join(parts[:3])

    def try_host(host: str) -> bool:
        try:
            sock = socket.socket(socket.AF_INET, socket.SOCK_STREAM)
            sock.settimeout(timeout_per_host)
            sock.connect((host, FSWIDGET_PORT))
            sock.sendall(FSWIDGET_REQUEST.encode("utf-8"))
            data = sock.recv(128)
            sock.close()
            msg = data.decode("utf-8", errors="ignore").strip()
            if msg.count("|") >= 7:
                return True
        except Exception:
            return False
        return False

    if try_host(local_ip):
        return local_ip

    for i in range(1, 255):
        candidate = f"{base_prefix}.{i}"
        if candidate == local_ip:
            continue
        if try_host(candidate):
            return candidate

    return None


def get_airport_info(server_url: str, api_key: str, icao_code: str) -> Optional[Dict[str, Any]]:
    """Get airport information from Abflug server API (v2)."""
    session = requests.Session()
    session.headers.update(
        {
            "X-API-Key": api_key,
            "Content-Type": "application/json",
        }
    )
    try:
        resp = session.get(
            f"{server_url.rstrip('/')}/api/airport",
            params={"icao_code": icao_code},
            timeout=5.0,
        )
        resp.raise_for_status()
        return resp.json()
    except Exception:
        return None


class LandingDetector:
    """Simplified state machine to detect the landing phase."""

    def __init__(self, airport_altitude_ft: float):
        self.airport_altitude_ft = airport_altitude_ft
        self.state = "cruise"
        self.last_sample_time: float = 0.0
        self.samples: List[ButterSample] = []
        self.touchdown_time: Optional[float] = None

    def add_sample(self, gps: GPSData, timestamp: Optional[float] = None) -> None:
        ts = timestamp if timestamp is not None else time.time()
        sample = ButterSample(
            timestamp=ts,
            altitude_ft=gps.altitude,
            vertical_speed_fpm=gps.vertical_speed,
            ground_speed_kts=gps.ground_speed,
        )
        self.samples.append(sample)
        cutoff = ts - 300.0
        self.samples = [s for s in self.samples if s.timestamp >= cutoff]
        self.last_sample_time = ts
        self._update_state(sample)

    def _update_state(self, s: ButterSample) -> None:
        vs = s.vertical_speed_fpm
        alt = s.altitude_ft

        cruise_vs_threshold = 100.0
        descending_vs_threshold = -200.0
        near_airport_margin_ft = 300.0

        if self.state in ("cruise", "unknown"):
            if vs < descending_vs_threshold:
                self.state = "descending"
        elif self.state == "descending":
            if alt <= self.airport_altitude_ft + near_airport_margin_ft and vs < 0:
                self.state = "landing"
        elif self.state == "landing":
            if (
                abs(vs) <= cruise_vs_threshold
                and abs(alt - self.airport_altitude_ft) <= near_airport_margin_ft
            ):
                self.state = "landed"
                self.touchdown_time = s.timestamp

    def has_landed(self) -> bool:
        return self.state == "landed" and self.touchdown_time is not None

    def get_analysis_window(self) -> Tuple[List[ButterSample], Optional[float]]:
        """Return samples in the analysis window."""
        if not self.has_landed():
            return [], None
        td = self.touchdown_time
        start = td - 10.0

        post_td_samples = sorted([s for s in self.samples if s.timestamp >= td], key=lambda s: s.timestamp)
        end_time = td + 15.0

        flat_vs_threshold = 1.0
        consecutive_flat_count = 0
        required_flat_samples = 20

        for sample in post_td_samples:
            if abs(sample.vertical_speed_fpm) <= flat_vs_threshold:
                consecutive_flat_count += 1
                if consecutive_flat_count >= required_flat_samples:
                    end_time = sample.timestamp
                    break
            else:
                consecutive_flat_count = 0

        window = [s for s in self.samples if start <= s.timestamp <= end_time]
        return window, td


def compute_butter_score(window: List[ButterSample], touchdown_time: float) -> ButterResult:
    """Compute butter score from samples in the analysis window."""
    if len(window) < 2:
        return ButterResult(
            score=50,
            smoothness=50,
            vertical_speed_fpm=0.0,
            ground_speed_kts=0.0,
            category="INSUFFICIENT DATA",
            emoji_line="🤔",
            message_line="Not enough data to rate this landing",
        )

    window = sorted(window, key=lambda s: s.timestamp)

    max_dv_dt = 0.0
    max_dg_dt = 0.0
    vs_at_touchdown = 0.0
    gs_at_touchdown = 0.0
    closest_td_sample: Optional[ButterSample] = None

    for i in range(1, len(window)):
        prev = window[i - 1]
        cur = window[i]
        dt = max(cur.timestamp - prev.timestamp, 1e-3)
        dv = cur.vertical_speed_fpm - prev.vertical_speed_fpm
        dg = cur.ground_speed_kts - prev.ground_speed_kts
        max_dv_dt = max(max_dv_dt, abs(dv) / dt)
        max_dg_dt = max(max_dg_dt, abs(dg) / dt)

        if closest_td_sample is None or abs(cur.timestamp - touchdown_time) < abs(
            closest_td_sample.timestamp - touchdown_time
        ):
            closest_td_sample = cur

    if closest_td_sample:
        vs_at_touchdown = closest_td_sample.vertical_speed_fpm
        gs_at_touchdown = closest_td_sample.ground_speed_kts

    dv_ref = 800.0
    dg_ref = 30.0
    dv_norm = min(max_dv_dt / dv_ref, 1.5)
    dg_norm = min(max_dg_dt / dg_ref, 1.5)

    roughness = 0.7 * dv_norm + 0.3 * dg_norm
    roughness_clamped = max(0.0, min(roughness, 1.5))

    base_score = int(round(100 * (1.0 - (roughness_clamped / 1.5))))

    vs_penalty = 0
    if vs_at_touchdown < -600:
        vs_penalty = -25
    elif vs_at_touchdown < -400:
        vs_penalty = -15
    elif vs_at_touchdown < -250:
        vs_penalty = -5
    elif vs_at_touchdown > -120:
        vs_penalty = -3

    final_score = max(0, min(100, base_score + vs_penalty))
    smoothness = final_score

    if final_score >= 90:
        category = "SILK SMOOTH TOUCHDOWN"
        emoji_line = "🧈🧈🧈"
        message_line = "BUTTER LANDING!"
    elif final_score >= 75:
        category = "NICE AND SOFT"
        emoji_line = "🧈"
        message_line = "Butter-ish landing"
    elif final_score >= 60:
        category = "MEH LANDING"
        emoji_line = "😐"
        message_line = "Meh, it'll pass"
    elif final_score >= 45:
        category = "SPICY ARRIVAL"
        emoji_line = "😬😬😬"
        message_line = "That was… firm"
    elif final_score >= 30:
        category = "CALL THE AMBULANCE"
        emoji_line = "🚑"
        message_line = "Cabin crew, prepare for meme review"
    else:
        category = "HULL LOSS"
        emoji_line = "💥"
        message_line = "HULL LOSS – send the clip"

    return ButterResult(
        score=final_score,
        smoothness=smoothness,
        vertical_speed_fpm=vs_at_touchdown,
        ground_speed_kts=gs_at_touchdown,
        category=category,
        emoji_line=emoji_line,
        message_line=message_line,
    )


def append_landing_to_config(
    config: Dict[str, Any],
    airport_icao: str,
    result: ButterResult,
) -> None:
    """Append landing result to butter.json under 'landings'."""
    landings = config.get("landings")
    if not isinstance(landings, list):
        landings = []
    landings.append(
        {
            "timestamp_utc": datetime.now(timezone.utc).strftime("%Y-%m-%dT%H:%M:%SZ"),
            "airport": airport_icao.upper(),
            "score": result.score,
            "smoothness": result.smoothness,
            "vertical_speed_fpm": result.vertical_speed_fpm,
            "ground_speed_kts": result.ground_speed_kts,
            "category": result.category,
        }
    )
    config["landings"] = landings
    save_config(config)


def build_discord_payload(
    airport_icao: str,
    result: ButterResult,
) -> Dict[str, Any]:
    title = f"Butter-o-meter – {airport_icao.upper()} landing"
    description = (
        f"{result.emoji_line} {result.message_line}\n"
        f"**Butter score**: {result.score}/100\n"
        f"**Smoothness**: {result.smoothness}/100\n"
        f"**Vertical speed**: {result.vertical_speed_fpm:.0f} fpm\n"
        f"**Ground speed**: {result.ground_speed_kts:.0f} kts\n"
        f"**Verdict**: {result.category}\n"
        f"Powered by Abflug"
    )

    return {
        "content": "",
        "embeds": [
            {
                "title": title,
                "description": description,
                "color": get_color_code("Aqua"),
            }
        ],
    }


class ButterMeterGUI:
    """Main GUI application for Butter-o-meter."""

    def __init__(self, root: tk.Tk):
        self.root = root
        self.root.title(f"{APP_NAME} v{APP_VERSION}")
        self.root.geometry("500x700")
        self.root.resizable(True, True)

        self.config = load_config()
        self.data_source: Optional[SimulatorDataSource] = None
        self.detector: Optional[LandingDetector] = None
        self.running = False
        self.result: Optional[ButterResult] = None
        self.current_airport_icao = ""

        self.setup_ui()

    def setup_ui(self):
        """Create the GUI layout."""
        # Main frame
        main_frame = ttk.Frame(self.root, padding="10")
        main_frame.pack(fill=tk.BOTH, expand=True)

        # Title
        title_label = tk.Label(
            main_frame,
            text="BUTTER-O-METER™",
            font=("Arial", 20, "bold"),
        )
        title_label.pack(pady=10)

        # Status indicator frame
        status_frame = ttk.Frame(main_frame)
        status_frame.pack(pady=5)

        status_label = tk.Label(status_frame, text="Simulator:", font=("Arial", 10))
        status_label.pack(side=tk.LEFT, padx=5)

        self.status_dot = tk.Canvas(status_frame, width=20, height=20, highlightthickness=0)
        self.status_dot.pack(side=tk.LEFT, padx=5)
        self.status_dot.create_oval(2, 2, 18, 18, fill="gray", outline="gray")
        self.status_dot_id = self.status_dot.find_all()[0]

        self.status_text = tk.Label(status_frame, text="Not connected", font=("Arial", 10))
        self.status_text.pack(side=tk.LEFT, padx=5)

        # Input frame
        input_frame = ttk.LabelFrame(main_frame, text="Configuration", padding="10")
        input_frame.pack(fill=tk.X, pady=10)

        # API Key
        ttk.Label(input_frame, text="API Key:").grid(row=0, column=0, sticky=tk.W, pady=5)
        self.api_key_var = tk.StringVar(value=self.config.get("api_key", ""))
        api_key_entry = ttk.Entry(input_frame, textvariable=self.api_key_var, width=40, show="*")
        api_key_entry.grid(row=0, column=1, pady=5, padx=5)

        api_key_help = tk.Label(
            input_frame,
            text="Get your API key at: https://abflug.cloud/login.html",
            font=("Arial", 8),
            fg="blue",
            cursor="hand2",
        )
        api_key_help.grid(row=1, column=1, sticky=tk.W, padx=5)
        api_key_help.bind("<Button-1>", lambda e: os.system(f'open "https://abflug.cloud/login.html"'))

        # Airport ICAO
        ttk.Label(input_frame, text="Airport ICAO:").grid(row=2, column=0, sticky=tk.W, pady=5)
        self.airport_var = tk.StringVar()
        airport_entry = ttk.Entry(input_frame, textvariable=self.airport_var, width=40)
        airport_entry.grid(row=2, column=1, pady=5, padx=5)

        # Control buttons
        button_frame = ttk.Frame(main_frame)
        button_frame.pack(fill=tk.X, pady=10)

        self.start_button = ttk.Button(
            button_frame, text="Start Monitoring", command=self.start_monitoring, width=20
        )
        self.start_button.pack(side=tk.LEFT, padx=5)

        self.stop_button = ttk.Button(
            button_frame, text="Stop", command=self.stop_monitoring, width=20, state=tk.DISABLED
        )
        self.stop_button.pack(side=tk.LEFT, padx=5)

        # Result display frame (initially hidden)
        self.result_frame = ttk.LabelFrame(main_frame, text="Result", padding="10")
        self.result_frame.pack(fill=tk.BOTH, expand=True, pady=10)

        self.result_canvas = tk.Canvas(
            self.result_frame, width=460, height=280, bg="white", highlightthickness=1, relief=tk.SUNKEN
        )
        self.result_canvas.pack(fill=tk.BOTH, expand=True)

        # Discord button (initially hidden)
        self.discord_button = ttk.Button(
            main_frame, text="Share to Discord", command=self.share_to_discord, state=tk.DISABLED
        )
        self.discord_button.pack(pady=5)

        # Powered by
        powered_label = tk.Label(
            main_frame, text="Powered by Abflug", font=("Arial", 8), fg="gray"
        )
        powered_label.pack(side=tk.BOTTOM, pady=5)

        # Initially hide result frame
        self.result_frame.pack_forget()
        self.discord_button.pack_forget()

    def update_status_dot(self, color: str, text: str):
        """Update the status indicator dot and text."""
        self.status_dot.itemconfig(self.status_dot_id, fill=color, outline=color)
        self.status_text.config(text=text)

    def start_monitoring(self):
        """Start monitoring for landing."""
        api_key = self.api_key_var.get().strip()
        airport_icao = self.airport_var.get().strip().upper()

        if not api_key:
            messagebox.showerror("Error", "Please enter your API key.")
            return

        if not airport_icao:
            messagebox.showerror("Error", "Please enter the landing airport ICAO code.")
            return

        # Save API key to config
        self.config["api_key"] = api_key
        if "server" not in self.config:
            self.config["server"] = DEFAULT_SERVER_URL
        save_config(self.config)

        self.current_airport_icao = airport_icao
        self.start_button.config(state=tk.DISABLED, text="Monitoring...")
        self.stop_button.config(state=tk.NORMAL)
        self.update_status_dot("yellow", "Connecting...")

        # Start monitoring in background thread
        thread = threading.Thread(target=self.monitoring_thread, daemon=True)
        thread.start()

    def stop_monitoring(self):
        """Stop monitoring."""
        self.running = False
        if self.data_source:
            self.data_source.stop()
        self.start_button.config(state=tk.NORMAL, text="Start Monitoring")
        self.stop_button.config(state=tk.DISABLED)
        self.update_status_dot("gray", "Stopped")

    def monitoring_thread(self):
        """Background thread for monitoring landing."""
        self.running = True

        try:
            # Fetch airport altitude
            server_url = self.config.get("server", DEFAULT_SERVER_URL)
            airport_info = get_airport_info(server_url, self.config["api_key"], self.current_airport_icao)

            if airport_info:
                airport_altitude_ft = float(airport_info.get("elevation_ft", 0.0) or 0.0)
            else:
                airport_altitude_ft = 0.0

            self.root.after(0, self.update_status_dot, "yellow", "Detecting simulator...")

            # Autodetect FSWidget
            fs_ip = discover_fswidget_ip()
            if not fs_ip:
                self.root.after(
                    0,
                    lambda: messagebox.showerror(
                        "Error", "Could not find FSWidget simulator endpoint.\nMake sure your sim is running with FSWidget enabled."
                    ),
                )
                self.root.after(0, lambda: self.start_button.config(state=tk.NORMAL, text="Start Monitoring"))
                self.root.after(0, self.stop_monitoring)
                return

            # Start data source
            self.data_source = SimulatorDataSource(
                source_type=DataSourceType.FSWIDGET,
                fswidget_ip=fs_ip,
                fswidget_port=FSWIDGET_PORT,
            )

            self.detector = LandingDetector(airport_altitude_ft=airport_altitude_ft)

            def on_gps_update(gps: GPSData) -> None:
                if self.detector:
                    self.detector.add_sample(gps)
                    # Update status dot to green when receiving data
                    self.root.after(0, self.update_status_dot, "green", "Receiving data...")

            self.data_source.on_gps_update = on_gps_update
            self.data_source.start()

            self.root.after(0, self.update_status_dot, "green", "Monitoring...")
            self.root.after(0, lambda: self.start_button.config(text="Monitoring..."))

            # Wait for landing
            touchdown_detected = False
            flat_vs_threshold = 1.0
            required_flat_samples = 20

            while self.running:
                time.sleep(0.5)
                if self.detector and self.detector.has_landed() and not touchdown_detected:
                    touchdown_detected = True
                    self.root.after(0, self.update_status_dot, "orange", "Touchdown detected...")
                    timeout = time.time() + 15.0

                if touchdown_detected and self.detector:
                    if self.detector.samples:
                        recent_samples = (
                            self.detector.samples[-required_flat_samples:]
                            if len(self.detector.samples) >= required_flat_samples
                            else self.detector.samples
                        )
                        post_td_samples = [
                            s for s in recent_samples if s.timestamp >= self.detector.touchdown_time
                        ]

                        if len(post_td_samples) >= required_flat_samples:
                            all_flat = all(
                                abs(s.vertical_speed_fpm) <= flat_vs_threshold
                                for s in post_td_samples[-required_flat_samples:]
                            )
                            if all_flat:
                                break

                    if time.time() >= timeout:
                        break

            if not self.running:
                return

            # Analyze landing
            if self.detector and self.detector.has_landed():
                window, td = self.detector.get_analysis_window()
                if window and td is not None:
                    self.result = compute_butter_score(window, td)
                    append_landing_to_config(self.config, self.current_airport_icao, self.result)
                    self.root.after(0, self.display_result)

            self.data_source.stop()

        except Exception as e:
            self.root.after(0, lambda: messagebox.showerror("Error", f"An error occurred: {str(e)}"))
            self.root.after(0, lambda: self.start_button.config(state=tk.NORMAL, text="Start Monitoring"))
            self.root.after(0, self.stop_monitoring)

    def display_result(self):
        """Display the butter-o-meter result."""
        if not self.result:
            return

        self.result_frame.pack(fill=tk.BOTH, expand=True, pady=10)
        self.discord_button.pack(pady=5)
        self.discord_button.config(state=tk.NORMAL)

        # Clear canvas
        self.result_canvas.delete("all")

        # Draw border
        self.result_canvas.create_rectangle(5, 5, 455, 350, outline="black", width=2)

        # Title
        self.result_canvas.create_text(
            230, 30, text="BUTTER-O-METER™", font=("Arial", 18, "bold"), fill="black"
        )

        # Emoji line
        self.result_canvas.create_text(230, 70, text=self.result.emoji_line, font=("Arial", 24))

        # Message line
        self.result_canvas.create_text(
            230, 110, text=self.result.message_line, font=("Arial", 14, "bold"), fill="black"
        )

        # Metrics
        y_pos = 160
        line_height = 30

        self.result_canvas.create_text(
            230, y_pos, text=f"Smoothness: {self.result.smoothness}/100",
            font=("Arial", 12),fill="black"
        )
        y_pos += line_height

        self.result_canvas.create_text(
            230,
            y_pos,
            text=f"Vertical Speed: {self.result.vertical_speed_fpm:.0f} fpm",
            font=("Arial", 12), 
            fill="black"
        )
        y_pos += line_height

        self.result_canvas.create_text(
            230,
            y_pos,
            text=f"Ground Speed: {self.result.ground_speed_kts:.0f} kts",
            font=("Arial", 12),
            fill="black"
        )
        y_pos += line_height * 1.5

        self.result_canvas.create_text(
            230, y_pos, text=self.result.category, font=("Arial", 12, "bold"), fill="black"
        )
        y_pos += line_height

        self.result_canvas.create_text(
            230, y_pos, text=f"Airport: {self.current_airport_icao.upper()}", font=("Arial", 12), fill="black"
        )

        self.update_status_dot("green", "Landing analyzed")

    def share_to_discord(self):
        """Share result to Discord."""
        if not self.result:
            return

        webhook_url = self.config.get("discord_webhook", "").strip()
        if not webhook_url:
            webhook_url = simpledialog.askstring(
                "Discord Webhook",
                "Enter Discord webhook URL:\n(starts with https://discord.com/api/webhooks/)",
            )
            if not webhook_url:
                return

            save_choice = messagebox.askyesno(
                "Save Webhook", "Save this webhook for next time?"
            )
            if save_choice:
                self.config["discord_webhook"] = webhook_url
                save_config(self.config)

        if not webhook_url.startswith("https://discord.com/api/webhooks/"):
            messagebox.showerror("Error", "Invalid Discord webhook URL.")
            return

        payload = build_discord_payload(self.current_airport_icao, self.result)
        try:
            resp = requests.post(webhook_url, json=payload, timeout=5.0)
            if resp.status_code in (200, 204):
                messagebox.showinfo("Success", "Discord webhook sent successfully!")
            else:
                messagebox.showerror("Error", f"Failed to send webhook: {resp.status_code}")
        except Exception as e:
            messagebox.showerror("Error", f"Error sending Discord webhook: {str(e)}")


def main():
    """Main entry point."""
    root = tk.Tk()
    app = ButterMeterGUI(root)
    root.mainloop()


if __name__ == "__main__":
    main()

