#!/usr/bin/env python3
"""
Abflug Client GUI v2 - Graphical interface for forwarding Aerofly GPS data

A redesigned GUI wrapper around abflug_client.py with a two-column layout.
"""

import tkinter as tk
from tkinter import ttk, font
import threading
import time
from typing import Optional
import sys
import os
import socket
import requests
import hashlib

# Import the AbflugClient class
sys.path.insert(0, os.path.dirname(os.path.abspath(__file__)))
from abflug_client_web import AbflugClient, DEFAULT_AEROFLY_PORT
from simulator_data_source import DataSourceType

# Constants
DEFAULT_SERVER_URL = "https://abflug.cloud"
STATUS_UPDATE_INTERVAL = 1000  # milliseconds
HEALTH_CHECK_INTERVAL = 5000  # milliseconds
APP_NAME = "Abflug Client with Webhooks (API v2.0)"
APP_VERSION = "2.0.0"


class AbflugClientGUI:
    """GUI application for Abflug Client."""
    
    def __init__(self, root: tk.Tk):
        self.root = root
        self.root.title(APP_NAME)
        self.root.geometry("800x900")
        self.root.resizable(True, True)
        
        # Client instance
        self.client: Optional[AbflugClient] = None
        self.client_thread: Optional[threading.Thread] = None
        self.is_connected = False

        # UDP monitoring (Aerofly) – independent from server connection
        self.udp_socket: Optional[socket.socket] = None
        self.udp_listener_thread: Optional[threading.Thread] = None
        self.udp_monitor_running: bool = False
        self.last_udp_time: float = 0.0
        
        # Status tracking
        self.last_health_check = 0
        self.health_status = "Unknown"
        self.last_api_key_check = 0
        self.api_key_valid = False
        self.server_online = False
        
        # Traffic polling for callsign retrieval
        self.own_icao: Optional[str] = None
        self.traffic_poll_thread: Optional[threading.Thread] = None
        self.traffic_poll_running = False
        self.server_callsign: Optional[str] = None
        self.traffic_poll_interval = 5.0  # seconds between traffic polls
        
        self.setup_ui()
        self.start_udp_monitor()
        self.update_status_loop()
    
    def setup_ui(self):
        """Set up the user interface with two-column layout."""
        # Main frame with padding
        main_frame = ttk.Frame(self.root, padding="10")
        main_frame.pack(fill="both", expand=True)
        
        # Title
        title_label = tk.Label(
            main_frame,
            text=f"{APP_NAME} v{APP_VERSION}",
            font=("Arial", 16, "bold")
        )
        title_label.pack(pady=(0, 10))
        
        # Create two-column layout
        content_frame = ttk.Frame(main_frame)
        content_frame.pack(fill="both", expand=True)
        
        # Left column - Configuration
        left_column = ttk.Frame(content_frame)
        left_column.pack(side="left", fill="both", expand=True, padx=(0, 10))
        
        # Right column - Status indicators
        right_column = ttk.Frame(content_frame)
        right_column.pack(side="right", fill="y", padx=(10, 0))
        
        # ========== LEFT COLUMN ==========
        
        # Server Configuration
        server_frame = ttk.LabelFrame(left_column, text="Server Configuration", padding="10")
        server_frame.pack(fill="x", pady=(0, 10))
        
        # Server URL
        ttk.Label(server_frame, text="Server URL:").grid(row=0, column=0, sticky="w", pady=5)
        self.server_url_entry = ttk.Entry(server_frame, width=30)
        self.server_url_entry.insert(0, DEFAULT_SERVER_URL)
        self.server_url_entry.grid(row=0, column=1, padx=5, pady=5, sticky="ew")
        
        # API Key
        ttk.Label(server_frame, text="API Key:").grid(row=1, column=0, sticky="w", pady=5)
        self.api_key_entry = ttk.Entry(server_frame, width=30, show="*")
        self.api_key_entry.grid(row=1, column=1, padx=5, pady=5, sticky="ew")
        
        # Callsign
        ttk.Label(server_frame, text="Callsign:").grid(row=2, column=0, sticky="w", pady=5)
        self.callsign_entry = ttk.Entry(server_frame, width=30)
        self.callsign_entry.grid(row=2, column=1, padx=5, pady=5, sticky="ew")
        ttk.Label(server_frame, text="(optional)", font=("Arial", 7), foreground="gray").grid(row=2, column=2, sticky="w", padx=5)
        
        server_frame.columnconfigure(1, weight=1)
        
        # Flight Plan
        flight_plan_frame = ttk.LabelFrame(left_column, text="Flight Plan", padding="10")
        flight_plan_frame.pack(fill="x", pady=(0, 10))
        
        # Departure and Arrival in a row
        ttk.Label(flight_plan_frame, text="Departure Airport:").grid(row=0, column=0, sticky="w", pady=5)
        self.departure_airport_entry = ttk.Entry(flight_plan_frame, width=15)
        self.departure_airport_entry.grid(row=0, column=1, padx=5, pady=5, sticky="w")
        
        ttk.Label(flight_plan_frame, text="Arrival:").grid(row=0, column=2, sticky="w", padx=(20, 5), pady=5)
        self.arrival_airport_entry = ttk.Entry(flight_plan_frame, width=15)
        self.arrival_airport_entry.grid(row=0, column=3, padx=5, pady=5, sticky="w")
        
        # Aircraft Model and Airline in a row
        ttk.Label(flight_plan_frame, text="Aircraft Model:").grid(row=1, column=0, sticky="w", pady=5)
        self.airplane_model_entry = ttk.Entry(flight_plan_frame, width=15)
        self.airplane_model_entry.grid(row=1, column=1, padx=5, pady=5, sticky="w")
        
        ttk.Label(flight_plan_frame, text="Airline:").grid(row=1, column=2, sticky="w", padx=(20, 5), pady=5)
        self.airplane_livery_entry = ttk.Entry(flight_plan_frame, width=15)
        self.airplane_livery_entry.grid(row=1, column=3, padx=5, pady=5, sticky="w")
        
        # Cruise Altitude
        ttk.Label(flight_plan_frame, text="Cruise Altitude:").grid(row=2, column=0, sticky="w", pady=5)
        self.cruise_altitude_entry = ttk.Entry(flight_plan_frame, width=15)
        self.cruise_altitude_entry.grid(row=2, column=1, padx=5, pady=5, sticky="w")
        ttk.Label(flight_plan_frame, text="(e.g., 30000 or FL300)", font=("Arial", 7), foreground="gray").grid(row=2, column=2, columnspan=2, sticky="w", padx=5)
        
        # Simulator Data Source
        simulator_frame = ttk.LabelFrame(left_column, text="Simulator Data Source", padding="10")
        simulator_frame.pack(fill="x", pady=(0, 10))
        
        self.data_source_var = tk.StringVar(value="udp")
        self.udp_radio = ttk.Radiobutton(
            simulator_frame,
            text="Aerofly UDP",
            variable=self.data_source_var,
            value="udp"
        )
        self.udp_radio.pack(anchor="w", pady=2)
        
        self.fswidget_radio = ttk.Radiobutton(
            simulator_frame,
            text="FSWidget TCP",
            variable=self.data_source_var,
            value="fswidget"
        )
        self.fswidget_radio.pack(anchor="w", pady=2)
        
        # FSWidget IP address field
        fswidget_frame = ttk.Frame(simulator_frame)
        fswidget_frame.pack(fill="x", pady=(5, 0))
        ttk.Label(fswidget_frame, text="Simulator IP:").pack(side="left", padx=5)
        self.fswidget_ip_entry = ttk.Entry(fswidget_frame, width=20)
        self.fswidget_ip_entry.insert(0, "0.0.0.0")
        self.fswidget_ip_entry.pack(side="left", padx=5)
        
        # Update FSWidget field state based on selection
        def update_fswidget_field_state(*args):
            if self.data_source_var.get() == "fswidget":
                self.fswidget_ip_entry.config(state="normal")
            else:
                self.fswidget_ip_entry.config(state="disabled")
        
        self.data_source_var.trace_add("write", update_fswidget_field_state)
        update_fswidget_field_state()  # Initial state
        
        # Discord webhook configuration
        discord_frame = ttk.LabelFrame(left_column, text="Discord Webhook", padding="10")
        discord_frame.pack(fill="x", pady=(0, 10))
        
        self.send_to_discord_var = tk.BooleanVar(value=False)
        self.send_to_discord_checkbox = tk.Checkbutton(discord_frame, text="Enable Discord webhooks", variable=self.send_to_discord_var)
        self.send_to_discord_checkbox.pack(anchor="w", pady=2)
        
        webhook_url_frame = ttk.Frame(discord_frame)
        webhook_url_frame.pack(fill="x", pady=2)
        ttk.Label(webhook_url_frame, text="Webhook URL:").pack(side="left", padx=5)
        self.discord_webhook_entry = ttk.Entry(webhook_url_frame, width=30)
        self.discord_webhook_entry.pack(side="left", padx=5, fill="x", expand=True)
        
        # ========== RIGHT COLUMN ==========
        
        # Status indicators frame
        status_indicators_frame = ttk.LabelFrame(right_column, text="Status", padding="10")
        status_indicators_frame.pack(fill="y", pady=(0, 10))
        
        # Server Status
        server_status_frame = ttk.Frame(status_indicators_frame)
        server_status_frame.pack(fill="x", pady=5)
        ttk.Label(server_status_frame, text="Server Status:").pack(side="left", padx=5)
        self.server_status_dot = tk.Canvas(server_status_frame, width=20, height=20, highlightthickness=0)
        self.server_status_dot.pack(side="left", padx=5)
        self.server_status_label = ttk.Label(server_status_frame, text="Not connected")
        self.server_status_label.pack(side="left", padx=5)
        
        # Aircraft Status
        aircraft_status_frame = ttk.Frame(status_indicators_frame)
        aircraft_status_frame.pack(fill="x", pady=5)
        ttk.Label(aircraft_status_frame, text="Aircraft Status:").pack(side="left", padx=5)
        self.aircraft_status_label = ttk.Label(aircraft_status_frame, text="Unknown", font=("Arial", 10, "bold"))
        self.aircraft_status_label.pack(side="left", padx=5)
        
        # Simulator Status
        simulator_status_frame = ttk.Frame(status_indicators_frame)
        simulator_status_frame.pack(fill="x", pady=5)
        self.simulator_status_label_text = ttk.Label(simulator_status_frame, text="Simulator Status:")
        self.simulator_status_label_text.pack(side="left", padx=5)
        self.udp_status_dot = tk.Canvas(simulator_status_frame, width=20, height=20, highlightthickness=0)
        self.udp_status_dot.pack(side="left", padx=5)
        self.udp_status_label = ttk.Label(simulator_status_frame, text="Waiting for data...")
        self.udp_status_label.pack(side="left", padx=5)
        
        # ========== BOTTOM SECTION ==========
        
        # Status Messages
        status_messages_frame = ttk.LabelFrame(main_frame, text="Status Messages", padding="10")
        status_messages_frame.pack(fill="both", expand=True, pady=(10, 10))
        
        # Create text widget with scrollbar
        text_frame = ttk.Frame(status_messages_frame)
        text_frame.pack(fill="both", expand=True)
        
        self.status_text = tk.Text(
            text_frame,
            height=8,
            width=80,
            wrap=tk.WORD,
            font=("Consolas", 9),
            state=tk.DISABLED
        )
        scrollbar = ttk.Scrollbar(text_frame, orient="vertical", command=self.status_text.yview)
        self.status_text.configure(yscrollcommand=scrollbar.set)
        
        self.status_text.pack(side="left", fill="both", expand=True)
        scrollbar.pack(side="right", fill="y")
        
        # Control buttons frame
        button_frame = ttk.Frame(main_frame)
        button_frame.pack(fill="x", pady=(0, 0))
        
        self.connect_button = ttk.Button(
            button_frame,
            text="Connect",
            command=self.toggle_connection,
            width=15
        )
        self.connect_button.pack(side="left", padx=5)
        
        ttk.Button(
            button_frame,
            text="Clear Log",
            command=self.clear_status,
            width=15
        ).pack(side="left", padx=5)
        
        # Initialize status dots
        self.update_udp_status(False, "Waiting for data...")
        self.update_server_status(False, "Not connected")
        self.aircraft_status_label.config(text="Not connected")
        
        # Add initial status message
        self.add_status_message("Ready. Enter API key and click Connect to start.")
    
    def update_udp_status(self, receiving: bool, message: str = ""):
        """Update UDP status indicator."""
        self.udp_status_dot.delete("all")
        
        if receiving:
            # Green dot
            self.udp_status_dot.create_oval(5, 5, 15, 15, fill="#00ff00", outline="#00aa00")
            status_text = "Receiving data" if not message else message
        else:
            # Gray dot
            self.udp_status_dot.create_oval(5, 5, 15, 15, fill="#888888", outline="#666666")
            status_text = "No data" if not message else message
        
        self.udp_status_label.config(text=status_text)
    
    def update_server_status(self, connected: bool, message: str = ""):
        """Update server connection status indicator."""
        self.server_status_dot.delete("all")
        
        if connected:
            # Green dot
            self.server_status_dot.create_oval(5, 5, 15, 15, fill="#00ff00", outline="#00aa00")
            status_text = "Connected" if not message else message
        else:
            # Red dot
            self.server_status_dot.create_oval(5, 5, 15, 15, fill="#ff0000", outline="#aa0000")
            status_text = "Disconnected" if not message else message
        
        self.server_status_label.config(text=status_text)
    
    def add_status_message(self, message: str):
        """Add a message to the status text box."""
        self.status_text.config(state=tk.NORMAL)
        timestamp = time.strftime("%H:%M:%S")
        self.status_text.insert(tk.END, f"[{timestamp}] {message}\n")
        self.status_text.see(tk.END)
        self.status_text.config(state=tk.DISABLED)
    
    def clear_status(self):
        """Clear the status text box."""
        self.status_text.config(state=tk.NORMAL)
        self.status_text.delete(1.0, tk.END)
        self.status_text.config(state=tk.DISABLED)

    def start_udp_monitor(self):
        """Start a background listener for Aerofly UDP, independent of server connection."""
        if self.udp_monitor_running:
            return

        try:
            self.udp_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            self.udp_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            if hasattr(socket, "SO_REUSEPORT"):
                self.udp_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEPORT, 1)
            self.udp_socket.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
            self.udp_socket.settimeout(1.0)
            # Listen on the same default Aerofly port used by the client
            self.udp_socket.bind(("", DEFAULT_AEROFLY_PORT))
        except Exception as e:
            self.add_status_message(f"Warning: Could not start Aerofly UDP monitor: {e}")
            if self.udp_socket:
                try:
                    self.udp_socket.close()
                except Exception:
                    pass
                self.udp_socket = None
            return

        self.udp_monitor_running = True

        def _udp_listener():
            while self.udp_monitor_running and self.udp_socket:
                try:
                    # We don't care about the data, just that something arrived
                    self.udp_socket.recvfrom(4096)
                    self.last_udp_time = time.time()
                except socket.timeout:
                    # Normal – just loop again
                    continue
                except OSError:
                    # Socket closed
                    break
                except Exception:
                    # Any other error: keep loop but don't spam logs
                    continue

        self.udp_listener_thread = threading.Thread(target=_udp_listener, daemon=True)
        self.udp_listener_thread.start()

    def stop_udp_monitor(self):
        """Stop the background Aerofly UDP listener."""
        self.udp_monitor_running = False
        if self.udp_socket:
            try:
                self.udp_socket.close()
            except Exception:
                pass
            self.udp_socket = None
    
    def check_server_health(self):
        """Check server health endpoint (API key validity is inferred from successful position updates)."""
        if not self.is_connected or not self.client:
            return
        
        current_time = time.time()
        # Only check every 5 seconds
        if current_time - self.last_health_check < (HEALTH_CHECK_INTERVAL / 1000):
            return
        
        self.last_health_check = current_time
        
        # Check server health (no API key required)
        # Note: API key validity is inferred from successful position updates sent by the client
        # No need to send redundant test PUT requests with zero coordinates
        try:
            response = self.client.http_session.get(
                f"{self.client.server_url}/api/health",
                timeout=5.0
            )
            if response.status_code == 200:
                data = response.json()
                self.health_status = f"OK (uptime: {data.get('uptime_seconds', 0):.0f}s)"
                self.server_online = True
                # If server is online and client is running, assume API key is valid
                # (it was validated at connection time, and ongoing position updates confirm it)
                self.api_key_valid = True
                self.update_server_status(True, self.health_status)
            else:
                self.health_status = f"Health check error: {response.status_code}"
                self.server_online = False
                self.update_server_status(False, self.health_status)
        except requests.exceptions.RequestException as e:
            # Check if response contains server logs
            if hasattr(e, 'response') and e.response is not None:
                try:
                    response_text = e.response.text[:200]
                    if "INFO:" in response_text:
                        self.add_status_message("ERROR: GET /api/health returned server logs in response!")
                except:
                    pass
            self.health_status = f"Server unreachable: {str(e)[:50]}"
            self.server_online = False
            self.update_server_status(False, self.health_status)
        except Exception as e:
            self.health_status = f"Server unreachable: {str(e)[:50]}"
            self.server_online = False
            self.update_server_status(False, self.health_status)
    
    def run_client(self):
        """Run the client in a separate thread."""
        try:
            # Start the client (this will block, so it's in a thread)
            # We need to override the start method to not block on the main loop
            self.start_client_non_blocking()
        except Exception as e:
            self.add_status_message(f"Client error: {e}")
            self.is_connected = False
            self.update_ui_state()
    
    def start_client_non_blocking(self):
        """Start client without the blocking main loop."""
        # Data source initialization is handled by AbflugClient.start()
        
        # Test server connection (health check - no API key needed)
        try:
            self.add_status_message("Making GET /api/health request...")
            test_response = self.client.http_session.get(
                f"{self.client.server_url}/api/health",
                timeout=5.0
            )
            if test_response.status_code == 200:
                self.add_status_message(f"Server is online (status: {test_response.status_code})")
                # Now test API key with a PUT request
                self.add_status_message("Making PUT /api/position request to validate API key...")
                test_payload = {
                    "latitude": 0.0,
                    "longitude": 0.0,
                    "altitude": 0.0,
                    "track": 0.0,
                    "ground_speed": 0.0,
                    "timestamp": time.time()
                }
                put_response = self.client.http_session.put(
                    f"{self.client.server_url}/api/position",
                    json=test_payload,
                    timeout=5.0
                )
                if put_response.status_code == 200:
                    self.add_status_message("API key validated successfully")
                    self.api_key_valid = True
                elif put_response.status_code == 401:
                    self.add_status_message("ERROR: Invalid API key (401 Unauthorized)")
                    self.add_status_message("Please check your API key and try again")
                    self.api_key_valid = False
                    # Disconnect immediately since API key is invalid
                    self.root.after(0, self.disconnect)
                    return
                else:
                    self.add_status_message(f"Warning: API key check returned status {put_response.status_code}")
                    self.api_key_valid = False
            else:
                self.add_status_message(f"Server health check failed: {test_response.status_code}")
                self.server_online = False
        except requests.exceptions.RequestException as e:
            # Identify which request failed by checking the error message
            error_str = str(e)
            if "health" in error_str.lower() or "GET" in error_str:
                request_type = "GET /api/health"
            elif "position" in error_str.lower() or "PUT" in error_str:
                request_type = "PUT /api/position"
            else:
                request_type = "unknown request"
            
            # Check if response content contains server logs
            if hasattr(e, 'response') and e.response is not None:
                try:
                    response_text = e.response.text[:500]  # First 500 chars
                    if "INFO:" in response_text or "HTTP" in response_text:
                        self.add_status_message(f"ERROR: Server returned logs in {request_type} response!")
                        self.add_status_message(f"Response preview: {response_text[:200]}")
                except:
                    pass
            
            self.add_status_message(f"Warning: {request_type} failed: {str(e)[:200]}")
            self.server_online = False
        
        # Start client (this will start data source and send thread)
        self.client.start()
        
        self.add_status_message("Client started successfully")
        if hasattr(self, 'client_source_type') and self.client_source_type == DataSourceType.FSWIDGET:
            self.add_status_message(f"Connecting to FSWidget at {self.client.data_source.fswidget_ip}:{self.client.data_source.fswidget_port}")
        else:
            self.add_status_message(f"Listening for Aerofly data on port {self.client.data_source.udp_port}")
        
        # Keep thread alive while running
        while self.client.running:
            time.sleep(1)
    
    def toggle_connection(self):
        """Connect or disconnect from the server."""
        if self.is_connected:
            # Disconnect
            self.disconnect()
        else:
            # Connect
            self.connect()
    
    def connect(self):
        """Connect to the server."""
        server_url = self.server_url_entry.get().strip()
        api_key = self.api_key_entry.get().strip()
        
        if not server_url:
            self.add_status_message("Error: Server URL is required")
            return
        
        if not api_key:
            self.add_status_message("Error: API Key is required")
            return
        
        # Check if user provided a callsign
        user_callsign = self.callsign_entry.get().strip() or None
        need_fetch_from_server = (user_callsign is None)
        
        # Only calculate ICAO if we need to fetch callsign from server
        if need_fetch_from_server:
            # Calculate own ICAO address (same method as server uses)
            hash_obj = hashlib.md5(api_key.encode())
            self.own_icao = hash_obj.hexdigest()[:6].upper()
        else:
            self.own_icao = None
        
        self.server_callsign = None
        
        # Disable inputs
        self.server_url_entry.config(state="disabled")
        self.api_key_entry.config(state="disabled")
        self.callsign_entry.config(state="disabled")
        self.departure_airport_entry.config(state="disabled")
        self.arrival_airport_entry.config(state="disabled")
        self.airplane_model_entry.config(state="disabled")
        self.airplane_livery_entry.config(state="disabled")
        self.cruise_altitude_entry.config(state="disabled")
        self.discord_webhook_entry.config(state="disabled")
        self.send_to_discord_checkbox.config(state="disabled")
        self.fswidget_ip_entry.config(state="disabled")
        self.udp_radio.config(state="disabled")
        self.fswidget_radio.config(state="disabled")
        self.connect_button.config(text="Connecting...", state="disabled")
        
        self.add_status_message(f"Connecting to {server_url}...")
        
        try:
            # Get data source selection
            use_fswidget = (self.data_source_var.get() == "fswidget")
            fswidget_ip = self.fswidget_ip_entry.get().strip() or "0.0.0.0"
            
            # Get flight plan data
            departure_airport = self.departure_airport_entry.get().strip() or None
            arrival_airport = self.arrival_airport_entry.get().strip() or None
            airplane_model = self.airplane_model_entry.get().strip() or None
            airplane_livery = self.airplane_livery_entry.get().strip() or None
            
            # Parse cruise altitude (handle FL notation)
            cruise_altitude = None
            cruise_alt_str = self.cruise_altitude_entry.get().strip()
            if cruise_alt_str:
                cruise_alt_str_upper = cruise_alt_str.upper()
                if cruise_alt_str_upper.startswith("FL"):
                    # FL notation: FL300 = 30000 ft, FL270 = 27000 ft, FL15 = 1500 ft
                    try:
                        fl_number = int(cruise_alt_str_upper[2:])
                        if fl_number < 0:
                            # Update GUI field to show absolute value
                            corrected_fl = abs(fl_number)
                            self.cruise_altitude_entry.config(state="normal")
                            self.cruise_altitude_entry.delete(0, tk.END)
                            self.cruise_altitude_entry.insert(0, f"FL{corrected_fl}")
                            self.cruise_altitude_entry.config(state="normal")
                            # Raise ValueError to stop processing
                            raise ValueError(f"Invalid FL notation '{cruise_alt_str}'. Must be positive. Corrected to FL{corrected_fl}")
                        cruise_altitude = fl_number * 100
                    except ValueError as e:
                        error_msg = str(e)
                        if "Invalid FL notation" in error_msg:
                            # This is our negative FL error
                            self.add_status_message(f"Error: {error_msg}")
                        else:
                            # This is a regular ValueError from int() conversion
                            self.add_status_message(f"Error: Invalid FL notation '{cruise_alt_str}'. Use format FL### (e.g., FL300)")
                        # Re-enable inputs
                        self.server_url_entry.config(state="normal")
                        self.api_key_entry.config(state="normal")
                        self.callsign_entry.config(state="normal")
                        self.departure_airport_entry.config(state="normal")
                        self.arrival_airport_entry.config(state="normal")
                        self.airplane_model_entry.config(state="normal")
                        self.airplane_livery_entry.config(state="normal")
                        self.cruise_altitude_entry.config(state="normal")
                        self.discord_webhook_entry.config(state="normal")
                        self.send_to_discord_checkbox.config(state="normal")
                        self.connect_button.config(text="Connect", state="normal")
                        return
                else:
                    # Direct feet value
                    try:
                        cruise_altitude = int(cruise_alt_str)
                        if cruise_altitude < 0:
                            # Update GUI field to show absolute value
                            self.cruise_altitude_entry.config(state="normal")
                            self.cruise_altitude_entry.delete(0, tk.END)
                            self.cruise_altitude_entry.insert(0, str(abs(cruise_altitude)))
                            self.cruise_altitude_entry.config(state="normal")
                            # Raise ValueError to stop processing
                            raise ValueError(f"Invalid cruise altitude '{cruise_alt_str}'. Must be a positive number. Corrected to {abs(cruise_altitude)}")
                    except ValueError as e:
                        error_msg = str(e)
                        if "Invalid cruise altitude" in error_msg:
                            # This is our negative number error
                            self.add_status_message(f"Error: {error_msg}")
                        else:
                            # This is a regular ValueError from int() conversion
                            self.add_status_message(f"Error: Invalid cruise altitude '{cruise_alt_str}'. Must be a number or FL notation (e.g., 30000 or FL300)")
                        # Re-enable inputs
                        self.server_url_entry.config(state="normal")
                        self.api_key_entry.config(state="normal")
                        self.callsign_entry.config(state="normal")
                        self.departure_airport_entry.config(state="normal")
                        self.arrival_airport_entry.config(state="normal")
                        self.airplane_model_entry.config(state="normal")
                        self.airplane_livery_entry.config(state="normal")
                        self.cruise_altitude_entry.config(state="normal")
                        self.discord_webhook_entry.config(state="normal")
                        self.send_to_discord_checkbox.config(state="normal")
                        self.connect_button.config(text="Connect", state="normal")
                        return
            
            # Get Discord webhook setting
            # Only use webhook URL if checkbox is enabled and URL is provided
            discord_webhook_url = None
            if self.send_to_discord_var.get():
                webhook_url = self.discord_webhook_entry.get().strip()
                if webhook_url:
                    # Validate webhook URL format
                    if not webhook_url.startswith("https://discord.com/api/webhooks/"):
                        self.add_status_message("Error: Invalid Discord webhook URL (must start with https://discord.com/api/webhooks/)")
                        # Re-enable inputs
                        self.server_url_entry.config(state="normal")
                        self.api_key_entry.config(state="normal")
                        self.callsign_entry.config(state="normal")
                        self.departure_airport_entry.config(state="normal")
                        self.arrival_airport_entry.config(state="normal")
                        self.airplane_model_entry.config(state="normal")
                        self.airplane_livery_entry.config(state="normal")
                        self.cruise_altitude_entry.config(state="normal")
                        self.discord_webhook_entry.config(state="normal")
                        self.send_to_discord_checkbox.config(state="normal")
                        self.connect_button.config(text="Connect", state="normal")
                        return
                    discord_webhook_url = webhook_url
                else:
                    self.add_status_message("Warning: Discord webhooks enabled but no URL provided. Webhooks will be disabled.")
            
            # Create client instance with user-provided callsign (if any)
            # Callsign will be sent via flight plan (highest precedence)
            self.client = AbflugClient(
                api_key=api_key,
                server_url=server_url,
                aerofly_port=DEFAULT_AEROFLY_PORT,
                callsign=user_callsign,  # Use user callsign if provided, None otherwise (sent via flight plan)
                use_fswidget=use_fswidget,
                fswidget_ip=fswidget_ip,
                send_to_discord=discord_webhook_url,
                departure_airport=departure_airport,
                arrival_airport=arrival_airport,
                airplane_model=airplane_model,
                airplane_livery=airplane_livery,
                cruise_altitude=cruise_altitude
            )
            
            # Set connected state BEFORE starting threads (so they see is_connected=True)
            self.is_connected = True
            
            # Start client in a separate thread
            self.client_thread = threading.Thread(target=self.run_client, daemon=True)
            self.client_thread.start()
            
            # Only start traffic polling if we need to fetch callsign from server
            if need_fetch_from_server:
                self.start_traffic_polling()
            else:
                self.add_status_message(f"Using user-provided callsign: {user_callsign}")
            
            self.add_status_message("Client started successfully")
            if use_fswidget:
                self.add_status_message(f"Connecting to FSWidget at {fswidget_ip}:58585")
            else:
                self.add_status_message(f"Listening for Aerofly data on port {DEFAULT_AEROFLY_PORT}")
            
        except Exception as e:
            self.add_status_message(f"Connection error: {e}")
            self.client = None
            self.is_connected = False
            self.own_icao = None
            # Re-enable inputs
            self.server_url_entry.config(state="normal")
            self.api_key_entry.config(state="normal")
            self.callsign_entry.config(state="normal")
            self.departure_airport_entry.config(state="normal")
            self.arrival_airport_entry.config(state="normal")
            self.airplane_model_entry.config(state="normal")
            self.airplane_livery_entry.config(state="normal")
            self.cruise_altitude_entry.config(state="normal")
            self.discord_webhook_entry.config(state="normal")
            self.send_to_discord_checkbox.config(state="normal")
            self.connect_button.config(text="Connect", state="normal")
            return
        
        # Update UI
        self.update_ui_state()
    
    def disconnect(self):
        """Disconnect from the server."""
        self.add_status_message("Disconnecting...")
        
        # Stop traffic polling
        self.stop_traffic_polling()
        
        if self.client:
            self.client.stop()
            self.client = None
        
        self.is_connected = False
        self.own_icao = None
        self.server_callsign = None
        self.update_callsign_display()
        self.update_ui_state()
        self.add_status_message("Disconnected")
    
    def update_ui_state(self):
        """Update UI elements based on connection state."""
        if self.is_connected:
            self.connect_button.config(text="Disconnect", state="normal")
            self.server_url_entry.config(state="disabled")
            self.api_key_entry.config(state="disabled")
            self.callsign_entry.config(state="disabled")
            self.departure_airport_entry.config(state="disabled")
            self.arrival_airport_entry.config(state="disabled")
            self.airplane_model_entry.config(state="disabled")
            self.airplane_livery_entry.config(state="disabled")
            self.cruise_altitude_entry.config(state="disabled")
            self.discord_webhook_entry.config(state="disabled")
            self.send_to_discord_checkbox.config(state="disabled")
            # Keep the selected data source, just disable the controls
            self.fswidget_ip_entry.config(state="disabled")
            # Disable radio buttons by making them unclickable (but keep selection visible)
            self.udp_radio.config(state="disabled")
            self.fswidget_radio.config(state="disabled")
        else:
            self.connect_button.config(text="Connect", state="normal")
            self.server_url_entry.config(state="normal")
            self.api_key_entry.config(state="normal")
            self.callsign_entry.config(state="normal")
            self.departure_airport_entry.config(state="normal")
            self.arrival_airport_entry.config(state="normal")
            self.airplane_model_entry.config(state="normal")
            self.airplane_livery_entry.config(state="normal")
            self.cruise_altitude_entry.config(state="normal")
            self.discord_webhook_entry.config(state="normal")
            self.send_to_discord_checkbox.config(state="normal")
            # Re-enable data source selection
            self.udp_radio.config(state="normal")
            self.fswidget_radio.config(state="normal")
            if self.data_source_var.get() == "fswidget":
                self.fswidget_ip_entry.config(state="normal")
            else:
                self.fswidget_ip_entry.config(state="disabled")
            self.update_server_status(False, "Not connected")
    
    def update_status_loop(self):
        """Periodically update status indicators."""
        # Update simulator status label based on data source
        if self.is_connected and self.client:
            if hasattr(self, 'client_source_type'):
                if self.client_source_type == DataSourceType.FSWIDGET:
                    self.simulator_status_label_text.config(text="FSWidget TCP:")
                else:
                    self.simulator_status_label_text.config(text="Aerofly UDP:")
        
        # Check simulator data status (UDP or FSWidget)
        if self.is_connected and self.client:
            if self.client.data_source.is_receiving_data(5.0):
                latest_gps = self.client.data_source.get_latest_gps()
                if latest_gps:
                    time_since_data = time.time() - self.client.data_source.last_gps_time
                    if time_since_data < 5.0:
                        self.update_udp_status(True, f"Receiving (last: {time_since_data:.1f}s ago)")
                    else:
                        self.update_udp_status(False, f"Stale data ({time_since_data:.1f}s old)")
                else:
                    self.update_udp_status(False, "No data received")
            else:
                self.update_udp_status(False, "No data received")
        elif self.last_udp_time > 0:
            # Fallback to UDP monitoring if client not connected
            time_since_udp = time.time() - self.last_udp_time
            if time_since_udp < 5.0:
                self.update_udp_status(True, f"Receiving (last: {time_since_udp:.1f}s ago)")
            else:
                self.update_udp_status(False, f"Stale data ({time_since_udp:.1f}s old)")
        else:
            self.update_udp_status(False, "No data received")

        # Server-related checks only when connected
        if self.is_connected and self.client:
            # Check server health
            self.check_server_health()
            # Update aircraft status
            try:
                aircraft_status_str = self.client.get_aircraft_status()
                if aircraft_status_str and isinstance(aircraft_status_str, str):
                    self.aircraft_status_label.config(text=aircraft_status_str.capitalize())
                else:
                    self.aircraft_status_label.config(text="Unknown")
            except Exception:
                self.aircraft_status_label.config(text="Unknown")
        else:
            self.aircraft_status_label.config(text="Not connected")
        
        # Schedule next update
        self.root.after(STATUS_UPDATE_INTERVAL, self.update_status_loop)
    
    def start_traffic_polling(self):
        """Start polling traffic API to retrieve callsign from server."""
        if self.traffic_poll_running:
            return
        
        self.traffic_poll_running = True
        self.traffic_poll_thread = threading.Thread(target=self._poll_traffic_for_callsign, daemon=True)
        self.traffic_poll_thread.start()
        self.add_status_message("Traffic polling started (will fetch callsign from server)")
    
    def stop_traffic_polling(self):
        """Stop polling traffic API."""
        self.traffic_poll_running = False
    
    def _poll_traffic_for_callsign(self):
        """Poll HTTP API for traffic to extract our own callsign from server response."""
        poll_count = 0
        
        while self.traffic_poll_running and self.is_connected and self.client:
            try:
                # Determine position to use: GPS data if available, otherwise use 0,0 (test coordinates)
                latest_gps = self.client.data_source.get_latest_gps()
                if latest_gps:
                    query_lat = latest_gps.latitude
                    query_lon = latest_gps.longitude
                    using_gps = True
                else:
                    # Use test coordinates (0,0) if no GPS data available
                    query_lat = 0.0
                    query_lon = 0.0
                    using_gps = False
                
                # Log first request attempt
                if poll_count == 0:
                    if using_gps:
                        self.root.after(0, lambda: self.add_status_message("Fetching callsign from server (using GPS position)..."))
                    else:
                        self.root.after(0, lambda: self.add_status_message("Fetching callsign from server (using test coordinates 0,0)..."))
                
                # Build API URL
                api_url = f"{self.client.server_url}/api/traffic"
                params = {
                    'lat': query_lat,
                    'lon': query_lon,
                    'radius_miles': 50.0  # Default radius
                }

                poll_count += 1
                
                # Debug: Log that we're making a request
                if poll_count == 1:
                    self.root.after(0, lambda: self.add_status_message(f"Making traffic API request (lat={query_lat:.4f}, lon={query_lon:.4f})"))
                
                # Make HTTP GET request
                response = self.client.http_session.get(api_url, params=params, timeout=5.0)
                response.raise_for_status()
                
                # Check if response contains server logs (should be JSON)
                try:
                    data = response.json()
                except ValueError as json_err:
                    # Response is not valid JSON - might contain server logs
                    response_text = response.text[:500]
                    if "INFO:" in response_text or "HTTP" in response_text:
                        self.root.after(0, lambda: self.add_status_message(f"ERROR: GET /api/traffic returned unexpected data. contact developer at support@abflug.cloud with the following information:"))
                        self.root.after(0, lambda: self.add_status_message(f" - Response preview: {response_text[:200]}"))
                    raise
                
                traffic_list = data.get('traffic', [])
                
                # Look for our own traffic (matching ICAO)
                found_own_traffic = False
                for traffic_dict in traffic_list:
                    if self.own_icao and traffic_dict.get('icao_address') == self.own_icao:
                        found_own_traffic = True
                        callsign = traffic_dict.get('callsign')
                        if callsign and callsign != self.server_callsign:
                            self.server_callsign = callsign
                            # Update callsign display and status message in GUI thread
                            self.root.after(0, lambda c=callsign: self.update_callsign_display())
                            self.root.after(0, lambda c=callsign: self.add_status_message(f"Callsign from server: {c}"))
                            # Stop polling once we've found and set the callsign
                            self.stop_traffic_polling()
                            self.root.after(0, lambda: self.add_status_message("Traffic polling stopped (callsign retrieved)"))
                            return  # Exit the thread cleanly
                        break
                
                # Log if we didn't find our own traffic (only first few times to avoid spam)
                if not found_own_traffic and poll_count <= 3:
                    self.root.after(0, lambda: self.add_status_message(f"Traffic query successful but own aircraft not found yet (poll #{poll_count})"))
                
            except requests.exceptions.RequestException as e:
                # Log first few errors for debugging
                if poll_count < 3:
                    error_msg = str(e)[:100]
                    self.root.after(0, lambda msg=error_msg: self.add_status_message(f"Traffic API error: {msg}"))
            except Exception as e:
                # Log unexpected errors
                if poll_count < 3:
                    error_msg = str(e)[:100]
                    self.root.after(0, lambda msg=error_msg: self.add_status_message(f"Unexpected error in traffic polling: {msg}"))
            
            # Wait before next poll
            time.sleep(self.traffic_poll_interval)
    
    def update_callsign_display(self):
        """Update the callsign entry field with server-provided callsign."""
        # Only update if we don't have a user-provided callsign
        # (This should only be called when fetching from server)
        if not self.callsign_entry.get().strip():
            self.callsign_entry.config(state="normal")
            self.callsign_entry.delete(0, tk.END)
            if self.server_callsign:
                self.callsign_entry.insert(0, self.server_callsign)
            self.callsign_entry.config(state="disabled")


def main():
    """Main entry point."""
    root = tk.Tk()
    app = AbflugClientGUI(root)
    root.mainloop()


if __name__ == '__main__':
    main()

