#!/usr/bin/env python3
"""
Abflug Client Sender - Forward Aerofly GPS Data to Multiplayer Server

This script receives UDP GPS data in foreflight format from Aerofly FS (or other simulator) and
forwards it to the Abflug multiplayer server via HTTP API. It acts as the "uplink" part of the multiplayer system.

Usage:
    python3 abflug_client.py --api-key YOUR_API_KEY --server SERVER_URL
"""

import socket
import threading
import time
import argparse
import re
import requests
from typing import Optional
from dataclasses import dataclass

# Constants
DEFAULT_AEROFLY_PORT = 49002  # Port to receive data from Aerofly
DEFAULT_SERVER_PORT = 80       # HTTP port (or 443 for HTTPS)
DEFAULT_SERVER_PATH = "/api/position"  # API endpoint for position updates
UPDATE_INTERVAL = 1.0  # Seconds between server updates (rate limiting)
HTTP_TIMEOUT = 5.0  # HTTP request timeout in seconds
# How long to wait without new simulator data before pausing sends (seconds)
STALE_GPS_TIMEOUT = 5.0
APP_VERSION = "1.0.0"
APP_NAME = "Abflug Client CLI"

@dataclass
class GPSData:
    """GPS data received from simulator."""
    latitude: float
    longitude: float
    altitude: float
    track: float
    ground_speed: float
    raw_message: str


class AbflugClient:
    """Client that forwards simulator GPS data to the multiplayer server."""
    
    def __init__(self, api_key: str, server_url: str, aerofly_port: int = DEFAULT_AEROFLY_PORT,
                 use_https: bool = False, callsign: Optional[str] = None):
        self.api_key = api_key
        self.callsign = callsign  # Optional user-provided callsign
        # Parse server URL (can be IP or domain, with or without port)
        if not server_url.startswith(('http://', 'https://')):
            protocol = 'https://' if use_https else 'http://'
            server_url = f"{protocol}{server_url}"
        # Ensure URL doesn't end with /
        self.server_url = server_url.rstrip('/')
        self.api_url = f"{self.server_url}{DEFAULT_SERVER_PATH}"
        self.aerofly_port = aerofly_port
        
        # State
        self.running = False
        self.latest_gps: Optional[GPSData] = None
        self.last_gps_time: float = 0.0
        self.stale_disconnect_sent: bool = False
        self.last_send_time = 0
        
        # Sockets (only for receiving from Aerofly)
        self.recv_socket: Optional[socket.socket] = None
        
        # HTTP session for connection reuse
        self.http_session = requests.Session()
        self.http_session.headers.update({
            'X-API-Key': self.api_key,
            'Content-Type': 'application/json'
        })
        
        # Threads
        self.receive_thread: Optional[threading.Thread] = None
        self.send_thread: Optional[threading.Thread] = None
        
        # Statistics
        self.messages_received = 0
        self.messages_sent = 0
        self.http_errors = 0
        self.start_time = time.time()
    
    def parse_gps_data(self, message: str) -> Optional[GPSData]:
        """Parse GPS data from Aerofly message.
        
        ForeFlight UDP protocol format:
        XGPS<simulator_name>,<longitude>,<latitude>,<altitude_msl>,<track_true_north>,<groundspeed_m/s>
        
        So the format is: XGPS...,longitude,latitude,altitude,track,ground_speed
        """
        # Match XGPS followed by optional simulator name and data
        pattern = r'XGPS(?:[^,]+)?,([-\d.]+),([-\d.]+),([-\d.]+),([-\d.]+),([-\d.]+)'
        match = re.match(pattern, message)
        if not match:
            return None
        
        # ForeFlight/Aerofly format: XGPS...,longitude,latitude,altitude,track,ground_speed
        longitude, latitude, altitude, track, ground_speed = map(float, match.groups())
        # Check for "menu state" condition (ignore)
        if (latitude == 0.0 and longitude == 0.0 and 
            altitude == 0.0 and track == 90.0 and ground_speed == 0.0):
            return None
        
        return GPSData(
            latitude=latitude,
            longitude=longitude,
            altitude=altitude,
            track=track,
            ground_speed=ground_speed,
            raw_message=message.strip()
        )
    
    def receive_loop(self) -> None:
        """Receive GPS data from Aerofly."""
        #print(f"Listening for Aerofly data on port {self.aerofly_port}...")
        
        while self.running:
            try:
                data, _ = self.recv_socket.recvfrom(1024)
                message = data.decode('utf-8')
                
                # Only process GPS messages
                if not message.startswith('XGPS'):
                    continue
                
                gps = self.parse_gps_data(message)
                if gps:
                    self.latest_gps = gps
                    self.last_gps_time = time.time()
                    self.stale_disconnect_sent = False
                    self.messages_received += 1
                    
            except socket.timeout:
                continue
            except Exception as e:
                if self.running:
                    print(f"Error receiving data: {e}")
    
    def send_loop(self) -> None:
        """Send GPS data to server via HTTP PUT at regular intervals."""
        #print(f"Forwarding to server {self.api_url}...")
        
        while self.running:
            try:
                # Check if we have data and enough time has passed
                current_time = time.time()
                # If GPS data is stale, pause sending and optionally disconnect once
                if self.latest_gps and (current_time - self.last_gps_time) > STALE_GPS_TIMEOUT:
                    if not self.stale_disconnect_sent:
                        # Send an explicit disconnect to the server
                        try:
                            self.http_session.delete(self.api_url, timeout=HTTP_TIMEOUT)
                            #print("Sent disconnect due to stale GPS data")
                        except requests.exceptions.RequestException:
                            # Ignore errors on disconnect attempt
                            pass
                        self.stale_disconnect_sent = True
                    # Stop sending until fresh data arrives
                    self.latest_gps = None
                    time.sleep(0.5)
                    continue

                if self.latest_gps and (current_time - self.last_send_time) >= UPDATE_INTERVAL:
                    # Prepare JSON payload
                    payload = {
                        "latitude": self.latest_gps.latitude,
                        "longitude": self.latest_gps.longitude,
                        "altitude": self.latest_gps.altitude,
                        "track": self.latest_gps.track,
                        "ground_speed": self.latest_gps.ground_speed,
                        "timestamp": current_time
                    }
                    # Add callsign if provided
                    if self.callsign:
                        payload["callsign"] = self.callsign
                    
                    # Send HTTP PUT request
                    try:
                        response = self.http_session.put(
                            self.api_url,
                            json=payload,
                            timeout=HTTP_TIMEOUT
                        )
                        response.raise_for_status()  # Raise exception for bad status codes
                        
                        self.messages_sent += 1
                        self.last_send_time = current_time
                        
                        #print(f"✓ Sent: {self.latest_gps.latitude:.4f}, {self.latest_gps.longitude:.4f}, "
                        #      f"{self.latest_gps.altitude:.0f}ft @ {self.latest_gps.ground_speed:.0f}kts")
                    
                    except requests.exceptions.RequestException as e:
                        self.http_errors += 1
                        if self.running:
                            #print(f"✗ HTTP error sending position: {e}")
                            if hasattr(e, 'response') and e.response is not None:
                                print(f"  Response: {e.response.status_code} - {e.response.text[:100]}")
                
                # Sleep a bit to avoid busy waiting
                time.sleep(0.1)
                
            except Exception as e:
                if self.running:
                    #print(f"Error in send loop: {e}")
                    time.sleep(1)  # Back off on error
    
    def start(self) -> None:
        """Start the client sender."""
        #print("=" * 60)
        #print("Winger Client Sender - Multiplayer Uplink (HTTP API)")
        #print("=" * 60)
        #print(f"Aerofly port: {self.aerofly_port}")
        #print(f"Server API: {self.api_url}")
        #print(f"Update rate: {1/UPDATE_INTERVAL:.1f} Hz")
        #print(f"API Key: {self.api_key[:8]}...")
        #print("=" * 60)
        
        # Create socket for receiving from Aerofly
        self.recv_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.recv_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        # Enable SO_REUSEPORT to allow multiple programs to bind to the same port
        # This allows both client_sender and rewinger to receive on port 49002
        if hasattr(socket, 'SO_REUSEPORT'):
            self.recv_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEPORT, 1)
        self.recv_socket.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
        self.recv_socket.settimeout(1.0)
        self.recv_socket.bind(('', self.aerofly_port))
        
        # Test server connection
        #print("Testing server connection...")
        try:
            test_response = self.http_session.get(
                f"{self.server_url}/api/health",
                timeout=HTTP_TIMEOUT
            )
            #print(f"✓ Server connection OK (status: {test_response.status_code})")
        except requests.exceptions.RequestException as e:
            print(f"⚠ Warning: Could not reach server health endpoint: {e}")
            print("  Continuing anyway - will retry on first position update...")
        
        # Start threads
        self.running = True
        self.receive_thread = threading.Thread(target=self.receive_loop, daemon=True)
        self.send_thread = threading.Thread(target=self.send_loop, daemon=True)
        # Note: No traffic receiving thread - rewinger receives directly via SO_REUSEPORT
        
        self.receive_thread.start()
        self.send_thread.start()
        
        #print("\nClient started successfully!")
        #print("Waiting for Aerofly data...")
        #print("Press Ctrl+C to stop\n")
        
        # Status updates
        try:
            while True:
                time.sleep(30)
                self.print_status()
        except KeyboardInterrupt:
            #print("\nStopping client...")
            self.stop()
    
    def stop(self) -> None:
        """Stop the client."""
        self.running = False
        
        if self.receive_thread:
            self.receive_thread.join(timeout=2)
        if self.send_thread:
            self.send_thread.join(timeout=2)
        
        if self.recv_socket:
            self.recv_socket.close()
        if self.http_session:
            self.http_session.close()
        
        #print("Client stopped")
    
    def print_status(self) -> None:
        """Print current status."""
        uptime = time.time() - self.start_time
        success_rate = (self.messages_sent / (self.messages_sent + self.http_errors) * 100) if (self.messages_sent + self.http_errors) > 0 else 0
        
        #print("\n" + "=" * 60)
        #print(f"Client Status - Uptime: {uptime/60:.1f} minutes")
        #print(f"Messages received (Aerofly): {self.messages_received}")
        #print(f"Messages sent (Server): {self.messages_sent}")
        #print(f"HTTP errors: {self.http_errors} (Success rate: {success_rate:.1f}%)")
        
        if self.latest_gps:
            print(f"Last position: {self.latest_gps.latitude:.4f}, {self.latest_gps.longitude:.4f}")
        else:
            print("No GPS data received yet")
        
        print("=" * 60 + "\n")


def main():
    """Main entry point."""
    parser = argparse.ArgumentParser(
        description="Abflug Client - Forward Aerofly data to multiplayer server"
    )
    parser.add_argument(
        '--api-key',
        required=True,
        help='Your API key for authentication'
    )
    parser.add_argument(
        '--server',
        required=True,
        help='Server URL (e.g., https://abflug.cloud, http://abflug.cloud, or your-server.com)'
    )
    parser.add_argument(
        '--use-https',
        action='store_true',
        help='Use HTTPS instead of HTTP (default: HTTP)'
    )
    parser.add_argument(
        '--aerofly-port',
        type=int,
        default=DEFAULT_AEROFLY_PORT,
        help=f'Port to receive Aerofly data (default: {DEFAULT_AEROFLY_PORT})'
    )
    parser.add_argument(
        '--callsign',
        type=str,
        default=None,
        help='Your callsign (optional, will use API key owner name if not provided)'
    )
    
    args = parser.parse_args()
    
    try:
        client = AbflugClient(
            api_key=args.api_key,
            server_url=args.server,
            aerofly_port=args.aerofly_port,
            use_https=args.use_https,
            callsign=args.callsign
        )
        client.start()
    except Exception as e:
        print(f"Fatal error: {e}")
        import traceback
        traceback.print_exc()
        return 1
    
    return 0


if __name__ == '__main__':
    exit(main())

