# Rewinger - Abflug.Cloud - Emanuele Bettoni
# Copyright (C) 2025  Abflug.Cloud - Emanuele Bettoni Development Team
# This program is under MIT License
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
# The above copyright notice and this permission notice shall be included in all
# copies or substantial portions of the Software.
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
# SOFTWARE.
#
# Rewinger is a modified version of:
# - Aircraft Tracker by Jugac64 / jlgabriel  from https://github.com/jlgabriel/Aerofly-FS4-Maps
# - Original Rewinger released by SageMage / Emanuele Bettoni from https://github.com/AeroSageMage/rewinger
# both released under MIT License by their respective authors
import socket
import threading
import re
import tkinter as tk
from tkintermapview import TkinterMapView
from tkinter import font as tkfont
from tkinter import filedialog
from PIL import Image, ImageTk
from typing import Optional, Dict, Any, Tuple, List
from dataclasses import dataclass
import time
import csv
import xml.etree.ElementTree as ET
import os
import requests
import hashlib

# Import shared simulator data source
from simulator_data_source import SimulatorDataSource, DataSourceType, GPSData as SimulatorGPSData



# Constants
UDP_PORT = 49002  # Port for direct simulator data (Aerofly, etc.)
TRAFFIC_PORT = 49004  # Port for multiplayer server traffic (legacy UDP, now using HTTP API)
WINDOW_SIZE = "1200x1000"
MAP_SIZE = (800, 600)
CONTROL_FRAME_WIDTH = 200
INFO_DISPLAY_SIZE = (24, 9)
UPDATE_INTERVAL = 1000  # milliseconds
RECEIVE_TIMEOUT = 5.0  # seconds
HTTP_POLL_INTERVAL = 2.0  # seconds between HTTP API polls for traffic (default, can be changed in GUI)
MIN_POLL_INTERVAL = 1.0  # minimum polling interval in seconds (to prevent server overload)
DEFAULT_TRAFFIC_RADIUS = 50.0  # miles - default radius for traffic query
SERVER_URL = "https://abflug.cloud"
APP_VERSION = "1.1.0-20251228"
APP_NAME = "Rewinger"
@dataclass
class GPSData:
    """Dataclass to store GPS data received from the flight simulator."""
    longitude: float
    latitude: float
    altitude: float
    track: float
    ground_speed: float

@dataclass
class AttitudeData:
    """Dataclass to store attitude data received from the flight simulator."""
    true_heading: float
    pitch: float
    roll: float
@dataclass    
class AircraftData:
    """Dataclass to store airplane data received from the network."""

    id: str # Unique identifier for this particular aircraft instance
    type_id: str # ID of the plane model from a predefined list/table
    registration: str # Official registration number assigned to the airplane by its national aviation authority
    callsign: str # Assigned radio call sign used by air traffic control and pilots for identification purposes
    icao24: str # International Civil Aviation Organization's unique four-character identifier for this aircraft
    FlightNumber: str

@dataclass
class AirTrafficData:
    """Dataclass to store traffic data received from the network."""
    icao_address: str
    latitude: float
    longitude: float
    altitude_ft: float
    vertical_speed_ft_min: float
    airborne_flag: int
    heading_true: float
    velocity_knots: float
    callsign: str
    


class UDPReceiver:
    """
    Class responsible for receiving and parsing UDP data from the flight simulator.
    Now uses SimulatorDataSource for GPS data (supports both UDP and FSWidget TCP).
    """
    def __init__(self, port: int = UDP_PORT, traffic_port: int = TRAFFIC_PORT,
                 use_fswidget: bool = False, fswidget_ip: str = "localhost"):
        self.port = port
        self.traffic_port = traffic_port
        self.socket: Optional[socket.socket] = None
        self.traffic_socket: Optional[socket.socket] = None
        
        # Use shared SimulatorDataSource for GPS data
        source_type = DataSourceType.FSWIDGET if use_fswidget else DataSourceType.UDP
        self.simulator_data_source = SimulatorDataSource(
            source_type=source_type,
            udp_port=port,
            fswidget_ip=fswidget_ip
        )
        # Set callback to update latest_gps_data when new GPS arrives
        self.simulator_data_source.on_gps_update = self._on_gps_update
        
        self.latest_gps_data: Optional[GPSData] = None
        self.latest_attitude_data: Optional[AttitudeData] = None
        self.latest_aircraft_data: Optional[AircraftData] = None
        self.traffic_data: Dict[str, Tuple[AirTrafficData, float]] = {}  # Store traffic data with timestamp
        self.running: bool = False
        self.receive_thread: Optional[threading.Thread] = None
        self.traffic_thread: Optional[threading.Thread] = None
        self.last_receive_time: float = 0
        self.log_to_csv: bool = False
        self.armed_for_recording: bool = False
        self.csv_files = {}
        self.simulator_name: str = "Unknown"
        self.simulator_name_set: bool = False
        
        # HTTP API settings for traffic
        self.use_http_api: bool = False
        self.api_server_url: Optional[str] = None
        self.api_key: Optional[str] = None
        self.callsign: Optional[str] = None  # Callsign from server (read-only, set by server based on client_sender)
        self.traffic_radius: float = DEFAULT_TRAFFIC_RADIUS
        self.poll_interval: float = HTTP_POLL_INTERVAL
        self.http_session: Optional[requests.Session] = None
        # Manual observation position (for observers without GPS/UDP data)
        self.manual_obs_lat: Optional[float] = None
        self.manual_obs_lon: Optional[float] = None
        # Own ICAO address (calculated from API key, same as server does)
        self.own_icao: Optional[str] = None
    
    def _on_gps_update(self, gps: SimulatorGPSData) -> None:
        """Callback when new GPS data arrives from SimulatorDataSource."""
        # Convert SimulatorGPSData to local GPSData format
        self.latest_gps_data = GPSData(
            longitude=gps.longitude,
            latitude=gps.latitude,
            altitude=gps.altitude,
            track=gps.track,
            ground_speed=gps.ground_speed
        )
        self.last_receive_time = time.time()


    def start_receiving(self) -> None:
        """Initialize and start the UDP receiving threads."""
        # Start simulator data source (handles GPS via UDP or FSWidget TCP)
        self.simulator_data_source.start()
        
        # Socket for attitude and aircraft data (only if using UDP mode)
        # If using FSWidget, we only get GPS data, so this socket is not needed for GPS
        # But we still need it for attitude/aircraft data if they come via UDP
        if self.simulator_data_source.source_type == DataSourceType.UDP:
            # Socket for direct simulator data (Aerofly, etc.) on port 49002
            # Note: SimulatorDataSource already binds to this port for GPS, but we need it
            # for attitude and aircraft data too, so we'll share the socket or use SO_REUSEPORT
            self.socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
            self.socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
            # Enable SO_REUSEPORT to allow multiple programs to bind to the same port
            if hasattr(socket, 'SO_REUSEPORT'):
                self.socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEPORT, 1)
            self.socket.setsockopt(socket.SOL_SOCKET, socket.SO_BROADCAST, 1)
            self.socket.settimeout(0.5)
            self.socket.bind(('', self.port))
            
            self.running = True
            self.receive_thread = threading.Thread(target=self._receive_data, daemon=True)
            self.receive_thread.start()
        else:
            # FSWidget mode: GPS comes from SimulatorDataSource, no UDP thread needed
            self.running = True
        
        # Legacy UDP traffic socket (kept for backward compatibility, but not used if HTTP API is enabled)
        self.traffic_socket = socket.socket(socket.AF_INET, socket.SOCK_DGRAM)
        self.traffic_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEADDR, 1)
        if hasattr(socket, 'SO_REUSEPORT'):
            self.traffic_socket.setsockopt(socket.SOL_SOCKET, socket.SO_REUSEPORT, 1)
        self.traffic_socket.settimeout(0.5)
        self.traffic_socket.bind(('', self.traffic_port))
        
        # Start traffic thread based on mode
        if self.use_http_api:
            self.traffic_thread = threading.Thread(target=self._poll_traffic_http, daemon=True)
        else:
            self.traffic_thread = threading.Thread(target=self._receive_traffic, daemon=True)
        
        self.traffic_thread.start()
    
    def set_http_api(self, server_url: str, api_key: str, radius: float = DEFAULT_TRAFFIC_RADIUS, 
                     callsign: Optional[str] = None, poll_interval: float = HTTP_POLL_INTERVAL,
                     manual_lat: Optional[float] = None, manual_lon: Optional[float] = None) -> None:
        """Configure HTTP API for traffic polling."""
        self.use_http_api = True
        self.api_server_url = server_url.rstrip('/')
        self.api_key = api_key
        # Callsign will be set from server response (not from parameter)
        self.callsign = None  # Will be updated from server when we receive our own traffic
        self.traffic_radius = radius
        # Enforce minimum polling interval
        self.poll_interval = max(poll_interval, MIN_POLL_INTERVAL)
        # Store manual observation position (for observers)
        self.manual_obs_lat = manual_lat
        self.manual_obs_lon = manual_lon
        
        # Calculate own ICAO address (same method as server uses)
        # This is used to identify our own aircraft in the traffic data
        # NOTE: REALLY IMPORTANT TO USE THE SAME METHOD AS THE SERVER USES!
        # IF CHANGED, THE SERVER WILL NOT BE ABLE TO IDENTIFY OUR OWN AIRCRAFT!
        # Changes needs to go in sync with the server!
        if api_key:
            hash_obj = hashlib.md5(api_key.encode())
            self.own_icao = hash_obj.hexdigest()[:6].upper()
        else:
            self.own_icao = None
        
        # Setup HTTP session
        if self.http_session:
            self.http_session.close()
        self.http_session = requests.Session()
        self.http_session.headers.update({
            'X-API-Key': self.api_key,
            'Content-Type': 'application/json'
        })
        
        # Stop old traffic thread if running
        #if self.traffic_thread and self.traffic_thread.is_alive():
        #    print("Stopping old traffic thread...")
            # The thread will exit when self.running becomes False, but we need to wait a bit
            # Actually, we can't easily stop it, so we'll just start a new one
            # The old one will continue but won't interfere
        
        # Start new HTTP polling thread
        if self.running:
            self.traffic_thread = threading.Thread(target=self._poll_traffic_http, daemon=True)
            self.traffic_thread.start()
            #print(f"HTTP API configured: {self.api_server_url} (radius: {radius} miles)")
            #print(f"Started HTTP API traffic polling thread")
    
    def disable_http_api(self) -> None:
        """Disable HTTP API and fall back to UDP."""
        # Send explicit disconnect to server if connected
        if self.use_http_api and self.http_session and self.api_server_url:
            try:
                disconnect_url = f"{self.api_server_url}/api/position"
                response = self.http_session.delete(disconnect_url, timeout=2.0)
                response.raise_for_status()
                #print("Sent explicit disconnect to server")
            except Exception as e:
                print(f"Failed to send disconnect (server will timeout): {e}")
        
        self.use_http_api = False
        if self.http_session:
            self.http_session.close()
            self.http_session = None
        
        # Clear manual observation position
        self.manual_obs_lat = None
        self.manual_obs_lon = None
        
        # Restart UDP traffic thread
        if self.running:
            if self.traffic_thread and self.traffic_thread.is_alive():
                # Old thread will continue, but we'll start UDP one
                pass
            self.traffic_thread = threading.Thread(target=self._receive_traffic, daemon=True)
            self.traffic_thread.start()
            #print("HTTP API disabled, using UDP for traffic")

    def _receive_data(self) -> None:
        """Continuously receive and parse UDP data while the thread is running."""
        while self.running:
            try:
                data, addr = self.socket.recvfrom(1024)
                self.last_receive_time = time.time()
                message = data.decode('utf-8')
                
                # Extract simulator name only from standard ForeFlight UDP messages
                if not self.simulator_name_set and ',' in message:
                    # Check for standard ForeFlight message types
                    if message.startswith(('XGPS', 'XATT', 'XTRAFFIC')):
                        parts = message.split(',', 1)
                        if len(parts) > 1:
                            msg_type = parts[0]
                            # Extract simulator name after the standard prefix
                            if msg_type.startswith('XGPS'):
                                self.simulator_name = msg_type[4:]
                            elif msg_type.startswith('XATT'):
                                self.simulator_name = msg_type[4:]
                            elif msg_type.startswith('XTRAFFIC'):
                                self.simulator_name = msg_type[8:]
                            
                            if self.simulator_name:  # Only set if we found a name
                                self.simulator_name_set = True
                                #print(f"Detected simulator: {self.simulator_name}")
                
                # GPS data is now handled by SimulatorDataSource, skip XGPS messages
                # (they're already being processed by the shared module)
                # if message.startswith('XGPS'):
                #     self.latest_gps_data = self._parse_gps_data(message)
                if message.startswith('XATT'):
                    self.latest_attitude_data = self._parse_attitude_data(message)
                if message.startswith('XAIRCRAFT'):
                    self.latest_aircraft_data = self._parse_aircraft_data(message)
                # Note: XTRAFFIC is now handled by _receive_traffic on port 49004
                        
                # Check if we need to start logging after arming
                if self.armed_for_recording and (self.latest_gps_data or len(self.traffic_data) > 0):
                    self.armed_for_recording = False
                    self.log_to_csv = True
                    #print("Recording automatically started after arming")
                    # Initialize CSV files
                    # Ensure output_recorder directory exists
                    os.makedirs("output_recorder", exist_ok=True)
                    timestamp = time.strftime("%Y%m%d-%H%M%S")
                    self.csv_files = {
                        'gps': open(f"output_recorder/output_GPS_DATA_{timestamp}.csv", "w", newline=''),
                        'attitude': open(f"output_recorder/output_ATTITUDE_DATA_{timestamp}.csv", "w", newline=''),
                        'traffic': open(f"output_recorder/output_TRAFFIC_DATA_{timestamp}.csv", "w", newline='')
                    }
                    # Write headers
                    csv.writer(self.csv_files['gps']).writerow(['Timestamp', 'Latitude', 'Longitude', 'Altitude', 'Track', 'Ground_Speed'])
                    csv.writer(self.csv_files['attitude']).writerow(['Timestamp', 'True_Heading', 'Pitch', 'Roll'])
                    csv.writer(self.csv_files['traffic']).writerow(['Timestamp', 'ICAO', 'Latitude', 'Longitude', 'Altitude_ft', 'VS_ft_min', 'Airborne', 'Heading', 'Velocity_kts', 'Callsign'])
            except socket.timeout:
                # This is expected, just continue the loop
                pass
            except Exception as e:
                print(f"Error receiving data: {e}")
    
    def _receive_traffic(self) -> None:
        """Continuously receive multiplayer traffic data from the server via UDP (legacy)."""
        #print("Started multiplayer traffic listener (UDP)")
        while self.running:
            try:
                data, addr = self.traffic_socket.recvfrom(1024)
                self.last_receive_time = time.time()
                message = data.decode('utf-8')
                
                # DEBUG: Show traffic messages
                #print(f"DEBUG TRAFFIC: Received from {addr}: {repr(message[:100])}")
                
                if message.startswith('XTRAFFIC'):
                    traffic_data = self._parse_traffic_data(message)
                    if traffic_data:
                        # Store with current timestamp
                        self.traffic_data[traffic_data.icao_address] = (traffic_data, time.time())
                        #print(f"DEBUG: Parsed traffic: {traffic_data.callsign} at {traffic_data.latitude},{traffic_data.longitude}")
                    else:
                        print(f"DEBUG: Failed to parse XTRAFFIC message!")
                        
            except socket.timeout:
                # This is expected, just continue the loop
                pass
            except Exception as e:
                if self.running:
                    print(f"Error receiving traffic data: {e}")
    
    def _poll_traffic_http(self) -> None:
        """Poll HTTP API for nearby traffic."""
        #print("Started HTTP API traffic polling")
        poll_count = 0
        while self.running:
            try:
                # Check if HTTP API is still enabled (might have been disabled)
                if not self.use_http_api:
                    print("HTTP API disabled, exiting polling thread")
                    break
                
                # Determine position to use for query: GPS data takes priority, fallback to manual observation position, then default
                query_lat = None
                query_lon = None
                using_manual_pos = False
                using_default_pos = False
                
                # Get GPS from SimulatorDataSource (works for both UDP and FSWidget)
                simulator_gps = self.simulator_data_source.get_latest_gps()
                print(f"DEBUG HTTP: Simulator GPS: {simulator_gps}")
                if simulator_gps:
                    # Use GPS data if available (normal operation)
                    query_lat = simulator_gps.latitude
                    query_lon = simulator_gps.longitude
                elif self.manual_obs_lat is not None and self.manual_obs_lon is not None:
                    # Use manual observation position if GPS is not available (observer mode)
                    query_lat = self.manual_obs_lat
                    query_lon = self.manual_obs_lon
                    using_manual_pos = True
                else:
                    # No position available - use default position (0,0) to allow traffic polling
                    # This allows traffic to be displayed even when GPS data is not yet available
                    # (e.g., when using FSWidget on mobile, or before simulator connects)
                    query_lat = 0.0
                    query_lon = 0.0
                    using_default_pos = True
                    if poll_count == 0:
                        print("DEBUG HTTP: No GPS data or manual position, using default position (0,0) for traffic query")
                
                # Check if HTTP session is set up
                if not self.http_session:
                    #print("ERROR: HTTP session not initialized!")
                    time.sleep(self.poll_interval)
                    continue
                
                # Build API URL
                api_url = f"{self.api_server_url}/api/traffic"
                params = {
                    'lat': query_lat,
                    'lon': query_lon,
                    'radius_miles': self.traffic_radius
                }
                
                poll_count += 1
                #if poll_count == 1:
                #    print(f"DEBUG HTTP: First poll to {api_url} with params {params}")
                
                # Make HTTP GET request
                response = self.http_session.get(api_url, params=params, timeout=5.0)
                response.raise_for_status()
                
                data = response.json()
                traffic_list = data.get('traffic', [])
                print(f"DEBUG HTTP: Received {len(traffic_list)} traffic aircraft from API")
                #print(traffic_list)
                current_time = time.time()
                new_traffic = {}
                for traffic_dict in traffic_list:
                    # Convert dict to AirTrafficData
                    traffic = AirTrafficData(
                        icao_address=traffic_dict['icao_address'],
                        latitude=traffic_dict['latitude'],
                        longitude=traffic_dict['longitude'],
                        altitude_ft=traffic_dict['altitude_ft'],
                        vertical_speed_ft_min=traffic_dict['vertical_speed_ft_min'],
                        airborne_flag=traffic_dict['airborne_flag'],
                        heading_true=traffic_dict['heading_true'],
                        velocity_knots=traffic_dict['velocity_knots'],
                        callsign=traffic_dict['callsign']
                    )
                    print(f"DEBUG HTTP: Traffic aircraft - ICAO: {traffic.icao_address}, Callsign: {traffic.callsign}, Pos: ({traffic.latitude:.4f}, {traffic.longitude:.4f})")
                    new_traffic[traffic.icao_address] = (traffic, current_time)
                    
                    # If this is our own traffic (same ICAO), extract callsign from server
                    if self.own_icao and traffic.icao_address == self.own_icao:
                        self.callsign = traffic.callsign
                        #print(f"DEBUG: Identified own traffic (ICAO: {self.own_icao}), callsign from server: {self.callsign}")
                
                # Update traffic_data (replace old data)
                self.traffic_data = new_traffic
                self.last_receive_time = current_time
                
                #if traffic_list:
                #    print(f"DEBUG HTTP: Received {len(traffic_list)} traffic aircraft from API")
                
            except requests.exceptions.RequestException as e:
                if self.running:
                    print(f"ERROR polling HTTP API for traffic: {e}")
                    if hasattr(e, 'response') and e.response is not None:
                        print(f"  Response status: {e.response.status_code}")
                        print(f"  Response text: {e.response.text[:200]}")
            except Exception as e:
                if self.running:
                    print(f"ERROR processing HTTP traffic data: {e}")
                    import traceback
                    traceback.print_exc()
            
            # Wait before next poll (use configured interval)
            time.sleep(self.poll_interval)
        
        #print("HTTP API traffic polling thread exited")
    
    @staticmethod
    def _parse_gps_data(message: str) -> Optional[GPSData]:
        """Parse GPS data from the received message."""
        # Match XGPS followed by optional simulator name and data
        pattern = r'XGPS(?:[^,]+)?,([-\d.]+),([-\d.]+),([-\d.]+),([-\d.]+),([-\d.]+)'
        match = re.match(pattern, message)
        if match:
            latitude, longitude, altitude, track, ground_speed = map(float, match.groups())
            
            # Check for the specific "menu state" condition
            if (latitude == 0.0 and longitude == 0.0 and 
                altitude == 0.0 and track == 90.0 and ground_speed == 0.0):
                # This is the menu state - return None instead
                return None
                
            # Otherwise return the valid GPS data
            return GPSData(*map(float, match.groups()))
        
        return None
    @staticmethod
    def _parse_attitude_data(message: str) -> Optional[AttitudeData]:
        """Parse attitude data from the received message."""
        pattern = r'XATT(?:[^,]+)?,([-\d.]+),([-\d.]+),([-\d.]+)'
        match = re.match(pattern, message)
        if match:
            return AttitudeData(*map(float, match.groups()))
        return None
    @staticmethod
    def _parse_aircraft_data(message: str) -> Optional[AircraftData]:
        """Parse Aircraft data from the received message."""
        pattern = r'^XAIRCRAFT(?:[^,]+)?,([A-Za-z0-9\-_]+),([A-Za-z0-9\-_]+),([A-Za-z0-9\-_]+),([A-Za-z0-9\-_]+),([A-Za-z0-9\-_]+),([A-Za-z0-9\-_]+)'
        match = re.match(pattern, message)
        if match:
            return AircraftData(*map(str, match.groups()))
        return None
    @staticmethod
    def _parse_traffic_data(message: str) -> Optional[AirTrafficData]:
        """Parse traffic data from the received message."""
        pattern = r'^XTRAFFIC(?:[^,]+)?,([A-Za-z0-9\-_]+),([-\d.]+),([-\d.]+),([-\d.]+),([-\d.]+),([01]),'\
                r'([-\d.]+),([-\d.]+),([A-Za-z0-9\-_\s]+)'
        match = re.match(pattern, message.strip())  # Strip whitespace before matching
        if match:
            groups = match.groups()
            return AirTrafficData(
                icao_address=str(groups[0]),
                latitude=float(groups[1]),
                longitude=float(groups[2]),
                altitude_ft=float(groups[3]),
                vertical_speed_ft_min=float(groups[4]),
                airborne_flag=int(groups[5]),
                heading_true=float(groups[6]),
                velocity_knots=float(groups[7]),
                callsign=str(groups[8])
            )
        return None

    def set_csv_logging(self, enabled: bool) -> None:
        """Enable or disable CSV logging."""
        # If we're turning off logging, close any open files
        if self.log_to_csv and not enabled:
            for file in self.csv_files.values():
                file.close()
            self.csv_files = {}
            
        self.log_to_csv = enabled
        self.armed_for_recording = False
        
        # If we're turning on logging, initialize new CSV files
        if enabled:
            # Ensure output_recorder directory exists
            os.makedirs("output_recorder", exist_ok=True)
            timestamp = time.strftime("%Y%m%d-%H%M%S")
            self.csv_files = {
                'gps': open(f"output_recorder/output_GPS_DATA_{timestamp}.csv", "w", newline=''),
                'attitude': open(f"output_recorder/output_ATTITUDE_DATA_{timestamp}.csv", "w", newline=''),
                'traffic': open(f"output_recorder/output_TRAFFIC_DATA_{timestamp}.csv", "w", newline='')
            }
            # Write headers
            csv.writer(self.csv_files['gps']).writerow(['Timestamp', 'Latitude', 'Longitude', 'Altitude', 'Track', 'Ground_Speed'])
            csv.writer(self.csv_files['attitude']).writerow(['Timestamp', 'True_Heading', 'Pitch', 'Roll'])
            csv.writer(self.csv_files['traffic']).writerow(['Timestamp', 'ICAO', 'Latitude', 'Longitude', 'Altitude_ft', 'VS_ft_min', 'Airborne', 'Heading', 'Velocity_kts', 'Callsign'])
        
        status = "enabled" if enabled else "disabled"
        #print(f"CSV logging {status}")
        
    def arm_recording(self) -> None:
        """Arm the recording system to start when data is received."""
        self.armed_for_recording = True
        self.log_to_csv = False
        #print("Recording armed and waiting for data")

    def get_latest_data(self) -> Dict[str, Any]:
        """Return the latest received GPS and attitude data."""
        # Clean outdated traffic data (older than 30 seconds)
        current_time = time.time()
        traffic_timeout = 30.0  # seconds
        self.traffic_data = {
            icao: (data, timestamp) 
            for icao, (data, timestamp) in self.traffic_data.items() 
            if current_time - timestamp < traffic_timeout
        }
        
        # Write to CSV if logging is enabled
        if self.log_to_csv and self.csv_files:
            current_time = time.time()
            
            # Write GPS data
            if self.latest_gps_data and 'gps' in self.csv_files:
                writer = csv.writer(self.csv_files['gps'])
                writer.writerow([
                    current_time,
                    self.latest_gps_data.latitude,
                    self.latest_gps_data.longitude,
                    self.latest_gps_data.altitude,
                    self.latest_gps_data.track,
                    self.latest_gps_data.ground_speed
                ])
                self.csv_files['gps'].flush()  # Ensure data is written immediately
            
            # Write attitude data
            if self.latest_attitude_data and 'attitude' in self.csv_files:
                writer = csv.writer(self.csv_files['attitude'])
                writer.writerow([
                    current_time,
                    self.latest_attitude_data.true_heading,
                    self.latest_attitude_data.pitch,
                    self.latest_attitude_data.roll
                ])
                self.csv_files['attitude'].flush()
            
            # Write traffic data
            if 'traffic' in self.csv_files:
                writer = csv.writer(self.csv_files['traffic'])
                for icao, (traffic_data, traffic_timestamp) in self.traffic_data.items():
                    writer.writerow([
                        traffic_timestamp,
                        traffic_data.icao_address,
                        traffic_data.latitude,
                        traffic_data.longitude,
                        traffic_data.altitude_ft,
                        traffic_data.vertical_speed_ft_min,
                        traffic_data.airborne_flag,
                        traffic_data.heading_true,
                        traffic_data.velocity_knots,
                        traffic_data.callsign
                    ])
                self.csv_files['traffic'].flush()
        
        return {
            'gps': self.latest_gps_data,
            'attitude': self.latest_attitude_data,
            'aircraft': self.latest_aircraft_data,
            'traffic': {icao: data for icao, (data, _) in self.traffic_data.items()},
            'connected': (time.time() - self.last_receive_time) < RECEIVE_TIMEOUT
        }

    def stop(self) -> None:
        """Stop receiving data."""
        self.running = False
        
        # Stop simulator data source
        self.simulator_data_source.stop()
        
        # Close sockets
        if self.socket:
            try:
                self.socket.close()
            except:
                pass
            self.socket = None
        
        if self.traffic_socket:
            try:
                self.traffic_socket.close()
            except:
                pass
            self.traffic_socket = None
        
        # Wait for threads to finish
        if self.receive_thread:
            self.receive_thread.join(timeout=2)
        if self.traffic_thread:
            self.traffic_thread.join(timeout=2)
        
        # Stop simulator data source
        self.simulator_data_source.stop()
        
        if self.socket:
            self.socket.close()
        if self.traffic_socket:
            self.traffic_socket.close()
        
        if self.http_session:
            self.http_session.close()
            self.http_session = None
        
        # Close any open CSV files
        if self.csv_files:
            for file in self.csv_files.values():
                file.close()

class AircraftTrackerApp:
    """
    Main application class for the Aircraft Tracker.
    Handles the GUI and updates the aircraft position on the map.
    """
    def __init__(self, master: tk.Tk):
        self.master = master
        self.master.title(f"{APP_NAME} / {APP_VERSION}")
        self.master.geometry(WINDOW_SIZE)
        
        # Initialize flight plan related attributes
        self.flight_plan_waypoints = []
        self.flight_plan_path = None
        self.current_kml_file = None
        
        self.setup_ui()
        # Initialize with default UDP (will be reconfigured if FSWidget is selected)
        self.data_source_type = DataSourceType.UDP
        self.fswidget_ip = "localhost"
        self.udp_receiver = UDPReceiver(use_fswidget=False, fswidget_ip="localhost")
        self.udp_receiver.start_receiving()
        self.setup_aircraft_marker()
        # Dictionary to keep track of traffic markers
        self.traffic_markers = {}
        # Setup a different icon for traffic
        self.traffic_image = Image.open("traffic_icon.png").resize((24, 24))
        self.update_aircraft_position()
        # Variables to track map center mode
        self.follow_aircraft = True
        self.map_center = None

    def setup_ui(self):
        """Set up the main user interface components."""
        self.main_frame = tk.Frame(self.master)
        self.main_frame.pack(fill="both", expand=True)

        # Create and configure the map widget
        self.map_widget = TkinterMapView(self.main_frame, width=MAP_SIZE[0], height=MAP_SIZE[1], corner_radius=0)
        self.map_widget.pack(side="left", fill="both", expand=True)

        # Create the control frame for additional UI elements
        self.control_frame = tk.Frame(self.main_frame, width=CONTROL_FRAME_WIDTH)
        self.control_frame.pack(side="right", fill="y")

        self.setup_map_selection()
        # Removed setup_info_display() to save space for multiplayer API frame

        # Add a connection status label
        self.connection_status = tk.Label(self.control_frame, text="Disconnected", fg="red")
        self.connection_status.pack(pady=5)

        # Add recording controls
        self.setup_recording_controls()
        
        # Add flight plan controls
        self.setup_flightplan_controls()

        # Add map control toggle
        self.setup_map_control()
        
        # Add HTTP API controls
        self.setup_http_api_controls()

        # Add a close button
        self.close_button = tk.Button(self.control_frame, text="Close Map", command=self.close_application)
        self.close_button.pack(side="bottom", pady=10)

        # Set up the window close protocol
        self.master.protocol("WM_DELETE_WINDOW", self.close_application)

    def setup_map_control(self):
        """Set up controls for map centering behavior."""
        map_control_frame = tk.Frame(self.control_frame)
        map_control_frame.pack(pady=5, fill="x")
        
        self.follow_var = tk.BooleanVar(value=True)
        self.follow_checkbox = tk.Checkbutton(
            map_control_frame, 
            text="Follow Aircraft", 
            variable=self.follow_var,
            command=self.toggle_follow_mode
        )
        self.follow_checkbox.pack(side="left", padx=5)
        
        # Proximity filter flag
        self.proximity_filter_var = tk.BooleanVar(value=True)
        self.proximity_filter_checkbox = tk.Checkbutton(
            map_control_frame,
            text="Filter Nearby Traffic",
            variable=self.proximity_filter_var
        )
        self.proximity_filter_checkbox.pack(side="left", padx=5)
        
    def toggle_follow_mode(self):
        """Toggle whether the map should automatically follow the aircraft."""
        self.follow_aircraft = self.follow_var.get()
        if not self.follow_aircraft:
            # Store current map center when disabling follow mode
            current_pos = self.map_widget.get_position()
            self.map_center = (current_pos[0], current_pos[1])
            #print(f"Follow mode disabled. Map center fixed at: {self.map_center}")
        else:
        # When re-enabling follow mode, if we have GPS data, immediately center on aircraft
            if self.udp_receiver.latest_gps_data:
                gps = self.udp_receiver.latest_gps_data
                self.map_widget.set_position(gps.latitude, gps.longitude)
                #print("Follow mode enabled. Centering on aircraft.")    
    
    def setup_http_api_controls(self):
        """Set up HTTP API controls for multiplayer traffic."""
        api_frame = tk.Frame(self.control_frame, relief=tk.GROOVE, bd=2)
        api_frame.pack(pady=5, padx=10, fill="x")
        
        # Title label
        tk.Label(api_frame, text="Multiplayer (HTTP API)", font=("Arial", 10, "bold")).pack(pady=(5,2))
        
        # Server URL
        tk.Label(api_frame, text="Server URL:", font=("Arial", 9)).pack(anchor=tk.W, padx=5, pady=2)
        self.api_server_url_entry = tk.Entry(api_frame, width=20, font=("Arial", 9))
        self.api_server_url_entry.pack(padx=5, pady=2, fill="x")
        self.api_server_url_entry.insert(0, SERVER_URL)
        
        # API Key
        tk.Label(api_frame, text="API Key:", font=("Arial", 9)).pack(anchor=tk.W, padx=5, pady=2)
        self.api_key_entry = tk.Entry(api_frame, width=20, font=("Arial", 9), show="*")
        self.api_key_entry.pack(padx=5, pady=2, fill="x")
        
        # Callsign (read-only, shows callsign from server)
        callsign_frame = tk.Frame(api_frame)
        callsign_frame.pack(padx=5, pady=2, fill="x")
        tk.Label(callsign_frame, text="Your Callsign:", font=("Arial", 9)).pack(anchor=tk.W, padx=5, pady=2)
        self.callsign_entry = tk.Entry(callsign_frame, width=20, font=("Arial", 9), state="readonly")
        self.callsign_entry.pack(padx=5, pady=2, fill="x")
        tk.Label(callsign_frame, text="(from server)", font=("Arial", 7), fg="gray").pack(anchor=tk.W, padx=5)
        
        # Simulator data source selection
        source_frame = tk.Frame(api_frame)
        source_frame.pack(padx=5, pady=5, fill="x")
        tk.Label(source_frame, text="Simulator Data Source:", font=("Arial", 9)).pack(anchor=tk.W, padx=5, pady=2)
        
        self.data_source_var = tk.StringVar(value="udp")
        udp_radio = tk.Radiobutton(
            source_frame,
            text="Aerofly UDP (default)",
            variable=self.data_source_var,
            value="udp",
            font=("Arial", 8)
        )
        udp_radio.pack(anchor=tk.W, padx=20)
        
        fswidget_radio = tk.Radiobutton(
            source_frame,
            text="FSWidget TCP",
            variable=self.data_source_var,
            value="fswidget",
            font=("Arial", 8)
        )
        fswidget_radio.pack(anchor=tk.W, padx=20)
        
        # FSWidget IP address field
        fswidget_frame = tk.Frame(source_frame)
        fswidget_frame.pack(fill="x", padx=20, pady=2)
        tk.Label(fswidget_frame, text="FSWidget IP:", font=("Arial", 8)).pack(side=tk.LEFT)
        self.fswidget_ip_entry = tk.Entry(fswidget_frame, width=15, font=("Arial", 8))
        self.fswidget_ip_entry.insert(0, "localhost")
        self.fswidget_ip_entry.pack(side=tk.LEFT, padx=5)
        
        # Update FSWidget field state based on selection
        def update_fswidget_field_state(*args):
            if self.data_source_var.get() == "fswidget":
                self.fswidget_ip_entry.config(state="normal")
            else:
                self.fswidget_ip_entry.config(state="disabled")
        
        self.data_source_var.trace("w", update_fswidget_field_state)
        update_fswidget_field_state()  # Initial state
        
        # Traffic radius and polling interval
        settings_frame = tk.Frame(api_frame)
        settings_frame.pack(padx=5, pady=2, fill="x")
        
        tk.Label(settings_frame, text="Radius (miles):", font=("Arial", 9)).pack(side=tk.LEFT)
        self.traffic_radius_entry = tk.Entry(settings_frame, width=8, font=("Arial", 9))
        self.traffic_radius_entry.pack(side=tk.LEFT, padx=5)
        self.traffic_radius_entry.insert(0, str(DEFAULT_TRAFFIC_RADIUS))
        
        tk.Label(settings_frame, text="Poll (sec):", font=("Arial", 9)).pack(side=tk.LEFT, padx=(10,0))
        self.poll_interval_entry = tk.Entry(settings_frame, width=6, font=("Arial", 9))
        self.poll_interval_entry.pack(side=tk.LEFT, padx=5)
        self.poll_interval_entry.insert(0, str(HTTP_POLL_INTERVAL))
        
        # Manual observation position (for observers without GPS/UDP)
        obs_frame = tk.Frame(api_frame)
        obs_frame.pack(padx=5, pady=2, fill="x")
        
        tk.Label(obs_frame, text="Observer Position (if no GPS):", font=("Arial", 9, "italic")).pack(anchor=tk.W, padx=5, pady=(5,2))
        
        obs_pos_frame = tk.Frame(obs_frame)
        obs_pos_frame.pack(padx=5, pady=2, fill="x")
        
        tk.Label(obs_pos_frame, text="Lat:", font=("Arial", 9)).pack(side=tk.LEFT)
        self.obs_lat_entry = tk.Entry(obs_pos_frame, width=10, font=("Arial", 9))
        self.obs_lat_entry.pack(side=tk.LEFT, padx=5)
        
        tk.Label(obs_pos_frame, text="Lon:", font=("Arial", 9)).pack(side=tk.LEFT, padx=(10,0))
        self.obs_lon_entry = tk.Entry(obs_pos_frame, width=10, font=("Arial", 9))
        self.obs_lon_entry.pack(side=tk.LEFT, padx=5)
        
        # Connect button
        self.api_connect_button = tk.Button(
            api_frame,
            text="Connect",
            font=("Arial", 9),
            command=self.connect_http_api,
            width=15
        )
        self.api_connect_button.pack(pady=5, padx=10)
        
        # Status label
        self.api_status_label = tk.Label(
            api_frame,
            text="Not connected",
            font=("Arial", 8),
            fg="gray"
        )
        self.api_status_label.pack(pady=2)

    def setup_flightplan_controls(self):
        """Set up flight plan loading and display controls."""
        fp_frame = tk.Frame(self.control_frame, relief=tk.GROOVE, bd=2)
        fp_frame.pack(pady=5, padx=10, fill="x")
        
        # Title label
        tk.Label(fp_frame, text="Flight Plan", font=("Arial", 10, "bold")).pack(pady=(5,2))
        
        # Load button
        self.load_kml_button = tk.Button(
            fp_frame,
            text="Load KML File",
            font=("Arial", 9),
            command=self.load_kml_file,
            width=15
        )
        self.load_kml_button.pack(pady=3, padx=10)
        
        # Toggle display checkbox
        self.show_flightplan_var = tk.BooleanVar(value=False)
        self.show_flightplan_checkbox = tk.Checkbutton(
            fp_frame, 
            text="Show Flight Plan", 
            variable=self.show_flightplan_var,
            command=self.toggle_flight_plan_display
        )
        self.show_flightplan_checkbox.pack(pady=3)
        
        # Status label
        self.flightplan_status = tk.Label(
            fp_frame, 
            text="No flight plan loaded",
            font=("Arial", 9)
        )
        self.flightplan_status.pack(pady=3)

    def load_kml_file(self):
        """Open a file dialog to select and load a KML file."""
        file_path = filedialog.askopenfilename(
            title="Select SimBrief KML File",
            filetypes=[("KML files", "*.kml"), ("All files", "*.*")]
        )
        
        if file_path:
            try:
                # Store the file path
                self.current_kml_file = file_path
                
                # Parse the KML file
                self.flight_plan_waypoints = self.parse_kml_file(file_path)
                
                if self.flight_plan_waypoints:
                    # Update status
                    self.flightplan_status.config(
                        text=f"Loaded: {os.path.basename(file_path)}",
                        fg="green"
                    )
                    
                    # Set the checkbox to checked and draw the flight plan
                    self.show_flightplan_var.set(True)
                    self.draw_flight_plan(self.flight_plan_waypoints)
                else:
                    self.flightplan_status.config(
                        text="Error: No route found in KML",
                        fg="red"
                    )
            except Exception as e:
                self.flightplan_status.config(
                    text=f"Error loading KML",
                    fg="red"
                )
                #print(f"Error loading KML file: {e}")

    def parse_kml_file(self, kml_file_path):
        """
        Parse a KML file and extract flight plan coordinates.
        
        Args:
            kml_file_path: Path to the KML file
            
        Returns:
            List of (latitude, longitude) tuples representing the flight plan route
        """
        try:
            # Parse the KML file
            tree = ET.parse(kml_file_path)
            root = tree.getroot()
            
            # Define the namespace
            namespace = {'kml': 'http://www.opengis.net/kml/2.2'}
            
            # Find all LineString elements which contain the flight path
            coordinates_elements = root.findall('.//kml:LineString/kml:coordinates', namespace)
            
            waypoints = []
            for coord_element in coordinates_elements:
                # KML coordinates are in lon,lat,alt format
                coord_text = coord_element.text.strip()
                for point in coord_text.split():
                    parts = point.split(',')
                    if len(parts) >= 2:
                        lon, lat = float(parts[0]), float(parts[1])
                        waypoints.append((lat, lon))  # Note: tkintermapview uses (lat, lon) order
            
            return waypoints
        except Exception as e:
            #print(f"Error parsing KML file: {e}")
            return []

    def draw_flight_plan(self, waypoints):
        """
        Draw the flight plan on the map.
        
        Args:
            waypoints: List of (latitude, longitude) tuples
        """
        if not waypoints:
            return
            
        # Create a path with the waypoints
        self.flight_plan_path = self.map_widget.set_path(waypoints, 
                                                        width=3,
                                                        color="#3080FF")
                                                        
        # Fit the map to show the entire flight plan
        if self.follow_aircraft:
            # If following aircraft, don't zoom out to fit flight plan
            pass
        else:
            # Otherwise, fit the map to show the entire flight plan
            self.map_widget.fit_bounds(waypoints)

    def toggle_flight_plan_display(self):
        """Toggle the display of the flight plan on the map."""
        show_plan = self.show_flightplan_var.get()
        
        if hasattr(self, 'flight_plan_path') and self.flight_plan_path:
            # Remove existing path
            self.flight_plan_path.delete()
            self.flight_plan_path = None
            
        if show_plan and hasattr(self, 'flight_plan_waypoints') and self.flight_plan_waypoints:
            # Redraw the path
            self.draw_flight_plan(self.flight_plan_waypoints)

    def setup_recording_controls(self):
        """Set up modern recording controls."""
        # Create a frame for recording controls
        recording_frame = tk.Frame(self.control_frame)
        recording_frame.pack(pady=10, fill="x")
        
        # Create variables to track button states
        self.record_var = tk.BooleanVar(value=False)
        self.armed_var = tk.BooleanVar(value=False)
        
        # Create a styled frame for buttons
        button_frame = tk.Frame(recording_frame, relief=tk.GROOVE, bd=2)
        button_frame.pack(pady=5, padx=10, fill="x")
        
        # Title label
        tk.Label(button_frame, text="Recording Controls", font=("Arial", 10, "bold")).pack(pady=(5,2))
        
        # Create the arming button
        self.arm_button = tk.Button(
            button_frame,
            text="ARM RECORDING",
            font=("Arial", 9),
            bg="#ff9900",  # Orange for armed state
            fg="black",
            activebackground="#ffcc00",
            relief=tk.RAISED,
            command=self.toggle_arm_recording,
            width=15
        )
        self.arm_button.pack(pady=3, padx=10)
        
        # Create the record button
        self.record_button = tk.Button(
            button_frame,
            text="START RECORDING",
            font=("Arial", 9),
            bg="#cccccc",  # Gray when not active
            fg="black",
            activebackground="#dddddd",
            relief=tk.RAISED,
            command=self.toggle_csv_logging,
            width=15
        )
        self.record_button.pack(pady=3, padx=10)
        
        # Create a status label
        self.recording_status = tk.Label(
            button_frame, 
            text="Status: Ready",
            font=("Arial", 9)
        )
        self.recording_status.pack(pady=5)

    def toggle_arm_recording(self):
        """Toggle the armed state for recording."""
        is_armed = not self.armed_var.get()
        self.armed_var.set(is_armed)
        
        # Update UI
        if is_armed:
            self.arm_button.config(
                bg="#ff9900",  # Orange
                text="ARMED",
                relief=tk.SUNKEN
            )
            self.record_button.config(
                bg="#cccccc",  # Gray
                text="START RECORDING",
                relief=tk.RAISED,
                state=tk.DISABLED
            )
            self.recording_status.config(text="Status: Armed for Recording", fg="orange")
            self.record_var.set(False)
            
            # Tell the UDP receiver to arm for recording
            self.udp_receiver.arm_recording()
        else:
            self.arm_button.config(
                bg="#dddddd",  # Light gray
                text="ARM RECORDING",
                relief=tk.RAISED
            )
            self.record_button.config(
                state=tk.NORMAL
            )
            self.recording_status.config(text="Status: Ready", fg="black")
            
            # Disarm recording
            self.udp_receiver.armed_for_recording = False

    def toggle_csv_logging(self):
        """Toggle CSV logging on or off and update button appearance."""
        # Don't allow toggling if armed
        if self.armed_var.get():
            return
            
        # Toggle the state
        is_logging = not self.record_var.get()
        self.record_var.set(is_logging)
        
        # Update buttons
        if is_logging:
            # Recording state
            self.record_button.config(
                bg="#ff3333",  # Red when recording
                text="STOP RECORDING",
                relief=tk.SUNKEN
            )
            self.arm_button.config(state=tk.DISABLED)
            self.recording_status.config(text="Status: Recording", fg="#ff3333")
        else:
            # Off state
            self.record_button.config(
                bg="#dddddd",  # Light gray
                text="START RECORDING",
                relief=tk.RAISED
            )
            self.arm_button.config(state=tk.NORMAL)
            self.recording_status.config(text="Status: Ready", fg="black")
        
        # Set CSV logging
        self.udp_receiver.set_csv_logging(is_logging)

    def setup_map_selection(self):
        """Set up the map selection listbox."""
        tk.Label(self.control_frame, text="Select Map:").pack(pady=(10, 5))

        listbox_frame = tk.Frame(self.control_frame)
        listbox_frame.pack(padx=0, pady=5)

        self.map_listbox = tk.Listbox(listbox_frame, width=24, height=6)
        self.map_listbox.pack(side="left")

        for option, _ in self.get_map_options():
            self.map_listbox.insert(tk.END, option)

        self.map_listbox.bind('<<ListboxSelect>>', lambda e: self.change_map())

    def setup_info_display(self):
        """Set up the information display area - REMOVED to save space."""
        # Removed to make room for multiplayer API frame
        self.info_display = None

    def setup_aircraft_marker(self):
        """Set up the aircraft marker image and related variables."""
        self.aircraft_image = Image.open("aircraft_icon.png").resize((32, 32))
        self.rotated_image = ImageTk.PhotoImage(self.aircraft_image)
        self.aircraft_marker = None
        self.initial_position_set = False

    def update_aircraft_position(self):
        """
        Update the aircraft's position on the map and the information display.
        This method is called periodically to refresh the display.
        """
        data = self.udp_receiver.get_latest_data()
        
        # DEBUG: Show connection state and traffic count
        #print(f"DEBUG UPDATE: connected={data['connected']}, traffic_count={len(data['traffic'])}, gps={data['gps'] is not None}")
        
        # Check if we're connected to the simulator
        if data['connected']:
            self.connection_status.config(text="Connected", fg="green")
            
            # Update API status label dynamically based on GPS availability
            if hasattr(self, 'api_status_label') and self.udp_receiver.use_http_api:
                if data['gps']:
                    # GPS data available - using normal mode
                    self.api_status_label.config(text="Connected (GPS Mode)", fg="green")
                elif (self.udp_receiver.manual_obs_lat is not None and 
                      self.udp_receiver.manual_obs_lon is not None):
                    # No GPS but manual position set - observer mode
                    self.api_status_label.config(text="Connected (Observer Mode)", fg="orange")
                else:
                    # Connected but waiting for position
                    self.api_status_label.config(text="Connected (Waiting for position)", fg="gray")
            
            # Update callsign display from server (if available)
            if hasattr(self, 'callsign_entry') and self.udp_receiver.callsign:
                self.callsign_entry.config(state="normal")
                self.callsign_entry.delete(0, tk.END)
                self.callsign_entry.insert(0, self.udp_receiver.callsign)
                self.callsign_entry.config(state="readonly")
            
            # Update traffic markers regardless of GPS data
            if data['traffic']:
                #print(f"DEBUG: Calling update_traffic_markers with {len(data['traffic'])} aircraft")
                self.update_traffic_markers(data['traffic'])
                
                # If we haven't set an initial position and we have traffic,
                # use the first traffic position to center the map
                if not self.initial_position_set and self.follow_aircraft:
                    first_traffic = next(iter(data['traffic'].values()))
                    self.map_widget.set_position(first_traffic.latitude, first_traffic.longitude)
                    self.map_widget.set_zoom(10)
                    self.initial_position_set = True
                    self.map_center = (first_traffic.latitude, first_traffic.longitude)
            # If we have GPS data, update the aircraft marker
            if data['gps'] and data['attitude']:
                self.update_aircraft_marker(data)
                # Removed update_info_display() call - info display removed to save space
        else:
            self.connection_status.config(text="Disconnected", fg="red")
            self.clear_info_display()
            
            # Keep traffic markers even when disconnected (just don't add new ones)
            # But clean up aircraft marker
            if self.aircraft_marker:
                self.aircraft_marker.delete()
                self.aircraft_marker = None

        # Check if armed recording should automatically start
        if self.armed_var.get() and not self.udp_receiver.armed_for_recording:
            # The UDPReceiver has detected data and auto-started recording
            if self.udp_receiver.log_to_csv:
                self.armed_var.set(False)
                self.record_var.set(True)
                self.arm_button.config(
                    bg="#dddddd",  # Light gray
                    text="ARM RECORDING",
                    relief=tk.RAISED,
                    state=tk.DISABLED
                )
                self.record_button.config(
                    bg="#ff3333",  # Red when recording
                    text="STOP RECORDING",
                    relief=tk.SUNKEN
                )
                self.recording_status.config(text="Status: Recording", fg="#ff3333")
            
        self.master.after(UPDATE_INTERVAL, self.update_aircraft_position)

    def clear_info_display(self):
        """Clear the information display when disconnected - REMOVED to save space."""
        # Info display removed to save space for multiplayer API frame
        pass

    def connect_http_api(self):
        """Connect to HTTP API for traffic polling."""
        server_url = self.api_server_url_entry.get().strip()
        api_key = self.api_key_entry.get().strip()
        
        if not server_url:
            self.api_status_label.config(text="Error: No server URL", fg="red")
            return
        
        if not api_key:
            self.api_status_label.config(text="Error: No API key", fg="red")
            return
        
        # Get data source selection
        use_fswidget = (self.data_source_var.get() == "fswidget")
        fswidget_ip = self.fswidget_ip_entry.get().strip() or "localhost"
        
        # Reconfigure UDPReceiver with selected data source if it changed
        if use_fswidget != (self.udp_receiver.simulator_data_source.source_type == DataSourceType.FSWIDGET):
            # Stop current receiver
            self.udp_receiver.stop()
            # Create new receiver with correct data source
            self.udp_receiver = UDPReceiver(use_fswidget=use_fswidget, fswidget_ip=fswidget_ip)
            self.udp_receiver.start_receiving()
            self.data_source_type = DataSourceType.FSWIDGET if use_fswidget else DataSourceType.UDP
            self.fswidget_ip = fswidget_ip
        
        try:
            radius = float(self.traffic_radius_entry.get() or DEFAULT_TRAFFIC_RADIUS)
        except ValueError:
            radius = DEFAULT_TRAFFIC_RADIUS
        
        try:
            poll_interval = float(self.poll_interval_entry.get() or HTTP_POLL_INTERVAL)
            # Enforce minimum
            if poll_interval < MIN_POLL_INTERVAL:
                self.api_status_label.config(text=f"Poll interval must be >= {MIN_POLL_INTERVAL}s", fg="orange")
                poll_interval = MIN_POLL_INTERVAL
                self.poll_interval_entry.delete(0, tk.END)
                self.poll_interval_entry.insert(0, str(MIN_POLL_INTERVAL))
        except ValueError:
            poll_interval = HTTP_POLL_INTERVAL
        
        # Callsign is now read-only and comes from server, so we don't read it from the entry
        # (The entry will be updated when we receive our own traffic from the server)
        
        # Get manual observation position (optional, for observers)
        manual_lat = None
        manual_lon = None
        obs_lat_str = self.obs_lat_entry.get().strip()
        obs_lon_str = self.obs_lon_entry.get().strip()
        
        if obs_lat_str and obs_lon_str:
            try:
                manual_lat = float(obs_lat_str)
                manual_lon = float(obs_lon_str)
                # Validate range
                if not (-90 <= manual_lat <= 90):
                    self.api_status_label.config(text="Error: Latitude must be -90 to 90", fg="red")
                    return
                if not (-180 <= manual_lon <= 180):
                    self.api_status_label.config(text="Error: Longitude must be -180 to 180", fg="red")
                    return
                #print(f"Using manual observation position: {manual_lat:.6f}, {manual_lon:.6f}")
            except ValueError:
                self.api_status_label.config(text="Error: Invalid lat/lon format", fg="red")
                return
        
        # Ensure URL has protocol
        if not server_url.startswith(('http://', 'https://')):
            server_url = f"http://{server_url}"
        
        # Configure UDPReceiver with HTTP API (this will start the polling thread)
        # Note: callsign parameter is no longer used - it will come from server response
        self.udp_receiver.set_http_api(server_url, api_key, radius, None, poll_interval, manual_lat, manual_lon)
        
        # Initial status - will be updated dynamically in update_aircraft_position()
        if manual_lat is not None and manual_lon is not None:
            # Manual position provided, but check if GPS is available
            if self.udp_receiver.latest_gps_data:
                self.api_status_label.config(text="Connected (GPS Mode)", fg="green")
            else:
                self.api_status_label.config(text="Connected (Observer Mode)", fg="orange")
        else:
            # No manual position - will use GPS if available
            if self.udp_receiver.latest_gps_data:
                self.api_status_label.config(text="Connected (GPS Mode)", fg="green")
            else:
                self.api_status_label.config(text="Connected (Waiting for GPS)", fg="gray")
        self.api_connect_button.config(text="Disconnect", command=self.disconnect_http_api)
    
    def disconnect_http_api(self):
        """Disconnect from HTTP API and fall back to UDP."""
        self.udp_receiver.disable_http_api()
        self.api_status_label.config(text="Not connected", fg="gray")
        self.api_connect_button.config(text="Connect", command=self.connect_http_api)
        # Clear callsign display
        if hasattr(self, 'callsign_entry'):
            self.callsign_entry.config(state="normal")
            self.callsign_entry.delete(0, tk.END)
            self.callsign_entry.config(state="readonly")
    
    def update_traffic_markers(self, traffic_data):
        """Update the traffic markers on the map."""
        # Get our own position if available (to filter out echoed traffic)
        own_lat = None
        own_lon = None
        if self.udp_receiver.latest_gps_data:
            own_lat = self.udp_receiver.latest_gps_data.latitude
            own_lon = self.udp_receiver.latest_gps_data.longitude
        
        # Remove markers for traffic that's no longer present
        for icao in list(self.traffic_markers.keys()):
            if icao not in traffic_data:
                self.traffic_markers[icao].delete()
                del self.traffic_markers[icao]
        
        # Debug: Show what traffic we have
        #if own_lat is not None:
        #    print(f"\nDEBUG MARKERS: Updating {len(traffic_data)} traffic markers (own pos: {own_lat:.4f}, {own_lon:.4f})")
        #else:
        #    print(f"\nDEBUG MARKERS: Updating {len(traffic_data)} traffic markers (no own GPS data)")
        
        # Update existing markers and add new ones
        for icao, data in traffic_data.items():
            # Determine if this traffic contact is "our" own aircraft (as identified by the server)
            is_own_aircraft = bool(self.udp_receiver.own_icao and icao == self.udp_receiver.own_icao)

            # Capture variables explicitly to avoid closure issues
            lat = data.latitude
            lon = data.longitude
            heading = data.heading_true
            callsign = data.callsign
            alt_ft = data.altitude_ft
            
            # For our own aircraft (identified via self.own_icao), use the same blue icon
            # as the local UDP aircraft marker instead of the orange traffic icon.
            if is_own_aircraft:
                rotated_image = self.rotate_image(heading)
            else:
                rotated_image = self.rotate_traffic_image(heading)
            
            if icao in self.traffic_markers:
                # Update existing marker
                self.traffic_markers[icao].delete()
            
            # Create new marker with callsign and altitude
            altitude_text = f"{int(alt_ft)}'"
            marker_text = f"{callsign} {altitude_text}" if callsign else altitude_text
            
            # Create marker with explicitly captured coordinates
            self.traffic_markers[icao] = self.map_widget.set_marker(
                lat, lon,
                icon=rotated_image,
                icon_anchor="center",
                text=marker_text
            )
            print(f"DEBUG MARKERS: Marker created for {callsign} at {lat:.6f}, {lon:.6f}")

    def rotate_traffic_image(self, angle: float) -> ImageTk.PhotoImage:
        """Rotate the traffic icon image by the given angle."""
        return ImageTk.PhotoImage(self.traffic_image.rotate(-angle))

    def update_aircraft_marker(self, data: Dict[str, Any]):
        """Update just the aircraft marker with the latest data."""
        gps_data: GPSData = data['gps']
        attitude_data: AttitudeData = data['attitude']
        aircraft_data: AircraftData = data['aircraft']
        
        if not self.initial_position_set:
            self.map_widget.set_position(gps_data.latitude, gps_data.longitude)
            self.map_widget.set_zoom(10)
            self.initial_position_set = True
            self.map_center = (gps_data.latitude, gps_data.longitude)
        self.rotated_image = self.rotate_image(attitude_data.true_heading)

        # Update or create the aircraft marker
        if self.aircraft_marker:
            self.aircraft_marker.delete()
            
        # Create marker with appropriate text
        # Priority: 1) Aircraft data callsign, 2) Server callsign (from multiplayer), 3) Simulator name
        marker_text = None
        if aircraft_data is not None:
            # Use aircraft data if available
            marker_text = aircraft_data.FlightNumber + " " + aircraft_data.callsign if aircraft_data.FlightNumber else aircraft_data.callsign
        elif self.udp_receiver.callsign:
            # Use callsign from server (multiplayer)
            marker_text = self.udp_receiver.callsign
        else:
            # Fallback to simulator name
            marker_text = self.udp_receiver.simulator_name
        
            self.aircraft_marker = self.map_widget.set_marker(
                gps_data.latitude, gps_data.longitude,
                icon=self.rotated_image,
                icon_anchor="center",
            text=marker_text
            )
        
        # Center map on aircraft if follow mode is enabled
        if self.follow_aircraft:
            self.map_widget.set_position(gps_data.latitude, gps_data.longitude)
        #elif self.map_center:
        #    # Stay on the fixed position when follow mode is disabled
        #    self.map_widget.set_position(self.map_center[0], self.map_center[1])

    def update_info_display(self, data: Dict[str, Any]):
        """Update the information display with the latest aircraft data - REMOVED to save space."""
        # Info display removed to save space for multiplayer API frame
        pass

    def rotate_image(self, angle: float) -> ImageTk.PhotoImage:
        """Rotate the aircraft icon image by the given angle."""
        return ImageTk.PhotoImage(self.aircraft_image.rotate(-angle))

    def change_map(self):
        """Change the map tile server based on the user's selection."""
        selected_indices = self.map_listbox.curselection()
        if selected_indices:
            _, tile_server = self.get_map_options()[selected_indices[0]]
            self.map_widget.set_tile_server(tile_server)

    @staticmethod
    def get_map_options() -> List[Tuple[str, str]]:
        """Return a list of available map options with their tile server URLs."""
        return [
            ("OpenStreetMap", "https://a.tile.openstreetmap.org/{z}/{x}/{y}.png"),
            ("OpenStreetMap DE", "https://tile.openstreetmap.de/tiles/osmde/{z}/{x}/{y}.png"),
            ("OpenStreetMap FR", "https://a.tile.openstreetmap.fr/osmfr/{z}/{x}/{y}.png"),
            ("OpenTopoMap", "https://a.tile.opentopomap.org/{z}/{x}/{y}.png"),
            ("Google Normal", "https://mt0.google.com/vt/lyrs=m&hl=en&x={x}&y={y}&z={z}"),
            ("Google Satellite", "https://mt0.google.com/vt/lyrs=s&hl=en&x={x}&y={y}&z={z}"),
            ("Google Terrain", "https://mt0.google.com/vt/lyrs=p&hl=en&x={x}&y={y}&z={z}"),
            ("Google Hybrid", "https://mt0.google.com/vt/lyrs=y&hl=en&x={x}&y={y}&z={z}"),
            ("Carto Dark Matter", "https://cartodb-basemaps-a.global.ssl.fastly.net/dark_all/{z}/{x}/{y}.png"),
            ("Carto Positron", "https://cartodb-basemaps-a.global.ssl.fastly.net/light_all/{z}/{x}/{y}.png"),
            ("ESRI World Imagery", "https://server.arcgisonline.com/ArcGIS/rest/services/World_Imagery/MapServer/tile/{z}/{y}/{x}"),
            ("ESRI World Street Map", "https://server.arcgisonline.com/ArcGIS/rest/services/World_Street_Map/MapServer/tile/{z}/{y}/{x}"),
            ("ESRI World Topo Map", "https://server.arcgisonline.com/ArcGIS/rest/services/World_Topo_Map/MapServer/tile/{z}/{y}/{x}")
        ]

    def close_application(self):
        """Clean up resources and close the application."""
        #print("Closing Aircraft Tracker...")
        
        # Send explicit disconnect if HTTP API is connected
        if self.udp_receiver.use_http_api:
            self.udp_receiver.disable_http_api()  # This will send disconnect
        
        # Clean up flight plan path if it exists
        if hasattr(self, 'flight_plan_path') and self.flight_plan_path:
            self.flight_plan_path.delete()
            
        self.udp_receiver.stop()
        self.master.destroy()

if __name__ == "__main__":
    #print("Starting Aircraft Tracker / Rewinger")
    #print(f"Listening for simulator data on port {UDP_PORT}")
    #print(f"Listening for multiplayer traffic on port {TRAFFIC_PORT}")
    root = tk.Tk()
    app = AircraftTrackerApp(root)
    root.mainloop()