#!/usr/bin/env python3
"""
Abflug Client Sender - Forward Aerofly GPS Data to Multiplayer Server

This script receives GPS data from flight simulators (UDP or FSWidget TCP) and
forwards it to the Abflug multiplayer server via HTTP API. It acts as the "uplink" part of the multiplayer system.

Usage:
    python3 abflug_client.py --api-key YOUR_API_KEY --server SERVER_URL
"""

import threading
import time
import argparse
import requests
from typing import Optional

# Import shared simulator data source
from simulator_data_source import SimulatorDataSource, DataSourceType, GPSData

# Constants
DEFAULT_AEROFLY_PORT = 49002  # Port to receive data from Aerofly
DEFAULT_SERVER_PORT = 80       # HTTP port (or 443 for HTTPS)
DEFAULT_SERVER_PATH = "/api/position"  # API endpoint for position updates
UPDATE_INTERVAL = 1.0  # Seconds between server updates (rate limiting)
HTTP_TIMEOUT = 5.0  # HTTP request timeout in seconds
# How long to wait without new simulator data before pausing sends (seconds)
STALE_GPS_TIMEOUT = 5.0
FSWIDGET_PORT = 58585  # Fixed port for FSWidget TCP protocol
APP_VERSION = "1.0.0"
APP_NAME = "Abflug Client CLI"


class AbflugClient:
    """Client that forwards simulator GPS data to the multiplayer server."""
    
    def __init__(self, api_key: str, server_url: str, aerofly_port: int = DEFAULT_AEROFLY_PORT,
                 use_https: bool = False, callsign: Optional[str] = None,
                 use_fswidget: bool = False, fswidget_ip: str = "localhost"):
        self.api_key = api_key
        self.callsign = callsign  # Optional user-provided callsign
        # Parse server URL (can be IP or domain, with or without port)
        if not server_url.startswith(('http://', 'https://')):
            protocol = 'https://' if use_https else 'http://'
            server_url = f"{protocol}{server_url}"
        # Ensure URL doesn't end with /
        self.server_url = server_url.rstrip('/')
        self.api_url = f"{self.server_url}{DEFAULT_SERVER_PATH}"
        
        # Initialize simulator data source
        source_type = DataSourceType.FSWIDGET if use_fswidget else DataSourceType.UDP
        self.data_source = SimulatorDataSource(
            source_type=source_type,
            udp_port=aerofly_port,
            fswidget_ip=fswidget_ip,
            fswidget_port=FSWIDGET_PORT
        )
        
        # State
        self.running = False
        self.stale_disconnect_sent: bool = False
        self.last_send_time = 0
        
        # HTTP session for connection reuse
        self.http_session = requests.Session()
        self.http_session.headers.update({
            'X-API-Key': self.api_key,
            'Content-Type': 'application/json'
        })
        
        # Threads
        self.send_thread: Optional[threading.Thread] = None
        
        # Statistics
        self.messages_sent = 0
        self.http_errors = 0
        self.start_time = time.time()
    
    
    def send_loop(self) -> None:
        """Send GPS data to server via HTTP PUT at regular intervals."""
        while self.running:
            try:
                # Get latest GPS data from data source
                latest_gps = self.data_source.get_latest_gps()
                current_time = time.time()
                
                # If GPS data is stale, pause sending and optionally disconnect once
                if latest_gps and not self.data_source.is_receiving_data(STALE_GPS_TIMEOUT):
                    if not self.stale_disconnect_sent:
                        # Send an explicit disconnect to the server
                        try:
                            self.http_session.delete(self.api_url, timeout=HTTP_TIMEOUT)
                        except requests.exceptions.RequestException:
                            # Ignore errors on disconnect attempt
                            pass
                        self.stale_disconnect_sent = True
                    # Stop sending until fresh data arrives
                    time.sleep(0.5)
                    continue

                if latest_gps and (current_time - self.last_send_time) >= UPDATE_INTERVAL:
                    # Reset stale disconnect flag when we have fresh data
                    self.stale_disconnect_sent = False
                    
                    # Prepare JSON payload
                    payload = {
                        "latitude": latest_gps.latitude,
                        "longitude": latest_gps.longitude,
                        "altitude": latest_gps.altitude,
                        "track": latest_gps.track,
                        "ground_speed": latest_gps.ground_speed,
                        "timestamp": current_time
                    }
                    # Add callsign if provided
                    if self.callsign:
                        payload["callsign"] = self.callsign
                    
                    # Send HTTP PUT request
                    try:
                        response = self.http_session.put(
                            self.api_url,
                            json=payload,
                            timeout=HTTP_TIMEOUT
                        )
                        response.raise_for_status()  # Raise exception for bad status codes
                        
                        self.messages_sent += 1
                        self.last_send_time = current_time
                    
                    except requests.exceptions.RequestException as e:
                        self.http_errors += 1
                        if self.running:
                            #print(f"✗ HTTP error sending position: {e}")
                            if hasattr(e, 'response') and e.response is not None:
                                print(f"  Response: {e.response.status_code} - {e.response.text[:100]}")
                
                # Sleep a bit to avoid busy waiting
                time.sleep(0.1)
                
            except Exception as e:
                if self.running:
                    #print(f"Error in send loop: {e}")
                    time.sleep(1)  # Back off on error
    
    def start(self) -> None:
        """Start the client sender."""
        # Start simulator data source
        self.data_source.start()
        
        # Test server connection
        try:
            test_response = self.http_session.get(
                f"{self.server_url}/api/health",
                timeout=HTTP_TIMEOUT
            )
        except requests.exceptions.RequestException as e:
            print(f"⚠ Warning: Could not reach server health endpoint: {e}")
            print("  Continuing anyway - will retry on first position update...")
        
        # Start send thread
        self.running = True
        self.send_thread = threading.Thread(target=self.send_loop, daemon=True)
        self.send_thread.start()
        
        #print("\nClient started successfully!")
        #print("Waiting for Aerofly data...")
        #print("Press Ctrl+C to stop\n")
        
        # Status updates
        try:
            while True:
                time.sleep(30)
                self.print_status()
        except KeyboardInterrupt:
            #print("\nStopping client...")
            self.stop()
    
    def stop(self) -> None:
        """Stop the client."""
        self.running = False
        
        # Stop simulator data source
        self.data_source.stop()
        
        if self.send_thread:
            self.send_thread.join(timeout=2)
        
        if self.http_session:
            self.http_session.close()
    
    def print_status(self) -> None:
        """Print current status."""
        uptime = time.time() - self.start_time
        success_rate = (self.messages_sent / (self.messages_sent + self.http_errors) * 100) if (self.messages_sent + self.http_errors) > 0 else 0
        
        latest_gps = self.data_source.get_latest_gps()
        if latest_gps:
            print(f"Last position: {latest_gps.latitude:.4f}, {latest_gps.longitude:.4f}")
        else:
            print("No GPS data received yet")
        
        print("=" * 60 + "\n")


def main():
    """Main entry point."""
    parser = argparse.ArgumentParser(
        description="Abflug Client - Forward Aerofly data to multiplayer server"
    )
    parser.add_argument(
        '--api-key',
        required=True,
        help='Your API key for authentication'
    )
    parser.add_argument(
        '--server',
        required=True,
        help='Server URL (e.g., https://abflug.cloud, http://abflug.cloud, or your-server.com)'
    )
    parser.add_argument(
        '--use-https',
        action='store_true',
        help='Use HTTPS instead of HTTP (default: HTTP)'
    )
    parser.add_argument(
        '--aerofly-port',
        type=int,
        default=DEFAULT_AEROFLY_PORT,
        help=f'Port to receive Aerofly data (default: {DEFAULT_AEROFLY_PORT})'
    )
    parser.add_argument(
        '--callsign',
        type=str,
        default=None,
        help='Your callsign (optional, will use API key owner name if not provided)'
    )
    parser.add_argument(
        '--use-fswidget',
        action='store_true',
        help='Use FSWidget TCP protocol instead of Aerofly UDP (default: UDP)'
    )
    parser.add_argument(
        '--fswidget-ip',
        type=str,
        default='localhost',
        help='IP address for FSWidget TCP connection (default: localhost)'
    )
    
    args = parser.parse_args()
    
    try:
        client = AbflugClient(
            api_key=args.api_key,
            server_url=args.server,
            aerofly_port=args.aerofly_port,
            use_https=args.use_https,
            callsign=args.callsign,
            use_fswidget=args.use_fswidget,
            fswidget_ip=args.fswidget_ip
        )
        client.start()
    except Exception as e:
        print(f"Fatal error: {e}")
        import traceback
        traceback.print_exc()
        return 1
    
    return 0


if __name__ == '__main__':
    exit(main())

